/*
 * (c) Copyright 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#ifdef ALSA
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/asoundlib.h>

#include "nwfiir.h"
#include "defs.h"
#include "alsaio.h"

static snd_pcm_t *pcm_output = NULL;
static snd_pcm_t *pcm_input = NULL;
static size_t framesize_output = 0;
static size_t framesize_input = 0;
static size_t fragsize_output = 0;
static size_t fragsize_input = 0;
static void *buf_output = NULL;

static int
sf2alsasf(struct sample_format *sf)
{
    int f = -1;
    
    switch (sf->bytes) {
    case 1:
	f = sf->is_signed ? SND_PCM_SFMT_S8: SND_PCM_SFMT_U8;
	break;
    case 2:
	if (sf->is_little_endian) {
	    f = sf->is_signed ? SND_PCM_SFMT_S16_LE: SND_PCM_SFMT_U16_LE;
	} else {
	    f = sf->is_signed ? SND_PCM_SFMT_S16_BE: SND_PCM_SFMT_U16_BE;
	}
	break;
    case 3:
	if (sf->is_little_endian) {
	    f = sf->is_signed ? SND_PCM_SFMT_S24_LE: SND_PCM_SFMT_U24_LE;
	} else {
	    f = sf->is_signed ? SND_PCM_SFMT_S24_BE: SND_PCM_SFMT_U24_BE;
	}
	break;
    case 4:
	if (sf->is_little_endian) {
	    f = sf->is_signed ? SND_PCM_SFMT_S32_LE: SND_PCM_SFMT_U32_LE;
	} else {
	    f = sf->is_signed ? SND_PCM_SFMT_S32_BE: SND_PCM_SFMT_U32_BE;
	}
	break;
    default:
    }
    return f;
}

static bool_t
init_device(snd_pcm_t **pcm,
	    int direction,
	    int fragsize,
	    int card,
	    int device,
	    int n_channels,
	    int rate,
	    struct sample_format *sf)
{
    snd_pcm_format_t format;
    snd_pcm_stream_params_t params;
    snd_pcm_stream_info_t info;
    snd_pcm_stream_setup_t setup;
    int err, n;

    switch (direction) {
    case SND_PCM_STREAM_CAPTURE:
	n = SND_PCM_OPEN_CAPTURE;
	break;
    case SND_PCM_STREAM_PLAYBACK:
	n = SND_PCM_OPEN_PLAYBACK;
	break;
    default:
	fprintf(stderr, "Internal error - incorrect direction value (%d)\n",
		direction);
	return false;
    }
    info.stream = direction;
    if ((err = snd_pcm_open(pcm, card, device, n)) < 0 ||
	(err = snd_pcm_stream_info(*pcm, &info)) < 0)
    {
	fprintf(stderr, "Audio open error: %s\n", snd_strerror(err));
	return false;
    }
    snd_pcm_stream_flush(*pcm, direction);
    bzero(&format, sizeof(snd_pcm_format_t));
    bzero(&params, sizeof(snd_pcm_stream_params_t));    
    format.interleave = 1;
    format.rate = rate;
    format.channels = n_channels;
    if ((format.format = sf2alsasf(sf)) == -1) {
	fprintf(stderr, "Sample format not compatible with ALSA\n");
	return false;
    }
    params.stream = direction;
    params.mode = SND_PCM_MODE_FRAGMENT;
    memcpy(&params.format, &format, sizeof(snd_pcm_format_t));
    params.start_mode = SND_PCM_START_DATA;
    params.xrun_mode = SND_PCM_XRUN_FLUSH;    
    params.frag_size = fragsize;
    params.buffer_size = (info.buffer_size == 0) ?
	    info.min_fragment_size * info.max_fragments : info.buffer_size;
    params.frames_min = 0;
    params.frames_xrun_max = 0;
    params.fill_mode = SND_PCM_FILL_SILENCE;
    params.frames_fill_max = fragsize;
    params.frames_xrun_max = 0;
    
    if ((err = snd_pcm_stream_params(*pcm, &params)) < 0) {
	fprintf(stderr, "Audio format set error: %s\n", snd_strerror(err));
	return false;
    }
    if ((err = snd_pcm_stream_prepare(*pcm, direction)) < 0) {
	fprintf(stderr, "Could not prepare audio: %s\n", snd_strerror(err));
	return false;
    }
    setup.stream = direction;
    if ((err = snd_pcm_stream_setup(*pcm, &setup)) < 0) {
	fprintf(stderr, "Could not get stream setup: %s\n", snd_strerror(err));
	return false;
    }
    if (fragsize != setup.frag_size) {
	fprintf(stderr, "Audio open error: requested fragment size (%d bytes) "
		"could not be met by hardware (%d bytes)\n",
		fragsize * sf->bytes * n_channels,
		setup.frag_size * sf->bytes * n_channels);
	return false;
    }
    return true;
}

bool_t
alsaio_getfragsize(int card,
		   int device,
		   bool_t is_input,
		   int *min_fragsize,
		   int *max_fragsize,
		   int *buffersize)
{
    snd_pcm_stream_info_t info;
    snd_pcm_t *pcm = NULL;
    int n;
    
    if (is_input) {
	n = SND_PCM_OPEN_CAPTURE;
	info.stream = SND_PCM_STREAM_CAPTURE;
    } else {
	n = SND_PCM_OPEN_PLAYBACK;
	info.stream = SND_PCM_STREAM_PLAYBACK;
    }
    if ((n = snd_pcm_open(&pcm, card, device, n)) < 0 ||
	(n = snd_pcm_stream_info(pcm, &info)) < 0)
    {
	fprintf(stderr, "Audio open error: %s\n", snd_strerror(n));
	return false;
    }    
    *min_fragsize = info.min_fragment_size;
    *max_fragsize = info.max_fragment_size;
    if (info.buffer_size == 0) {
	    *buffersize = info.min_fragment_size * info.max_fragments;
    } else {
	    *buffersize = info.buffer_size;
    }
    snd_pcm_close(pcm);
    return true;
}

bool_t
alsaio_init_output(int fragsize,
		   int card,
		   int device,
		   int n_channels,
		   int rate,
		   struct sample_format *sf)
{
    framesize_output = sf->bytes * n_channels;
    fragsize /= framesize_output;
    if (!init_device(&pcm_output, SND_PCM_STREAM_PLAYBACK, fragsize,
		     card, device, n_channels, rate, sf))
    {
	return false;
    }
    fragsize_output = fragsize;
    if ((buf_output = malloc(fragsize_output * framesize_output)) == NULL) {
	fprintf(stderr, "Could not allocate output buffer space\n");
	return false;
    }
    bzero(buf_output, fragsize_output);
    return true;
}

bool_t
alsaio_init_input(int fragsize,
		  int n_frags,
		  int card,
		  int device,
		  int n_channels,
		  int rate,
		  struct sample_format *sf)
{
    framesize_input = sf->bytes * n_channels;	
    fragsize /= framesize_input;
    if (!init_device(&pcm_input, SND_PCM_STREAM_CAPTURE, fragsize, 
		     card, device, n_channels, rate, sf))
    {
	return false;
    }
    fragsize_input = n_frags * fragsize;
    return true;
}

void
alsaio_output(void *buf,
	      int size)
{
    int err, n, i;

    size /= framesize_output;
    if ((n = size % fragsize_output) != 0) {
	i = size;
	size -= n;
	if (i != n) {
	    if ((err = snd_pcm_write(pcm_output, buf, size)) < 0) {
		fprintf(stderr, "Audio output error: %s\n", snd_strerror(err));
		fprintf(stderr, "Aborting!\n");
		nwfiir_abort_all_processes();
	    }
	}
	memcpy(buf_output, buf + size, n);
	if ((err = snd_pcm_write(pcm_output, buf_output,
				 fragsize_output)) < 0)
	{
	    fprintf(stderr, "Audio output error: %s\n", snd_strerror(err));
	    fprintf(stderr, "Aborting!\n");
 	    nwfiir_abort_all_processes();
	}
    } else {
	if ((err = snd_pcm_write(pcm_output, buf, size)) < 0) {
	    fprintf(stderr, "Audio output error: %s\n", snd_strerror(err));
	    fprintf(stderr, "Aborting!\n");
	    nwfiir_abort_all_processes();
	}
    }
}

int
alsaio_input(void *buf)
{
    int err;

    if ((err = snd_pcm_read(pcm_input, buf, fragsize_input)) < 0) {
	fprintf(stderr, "Audio input error: %s\n", snd_strerror(err));
	fprintf(stderr, "Aborting!\n");
	nwfiir_abort_all_processes();
    }
    return fragsize_input * framesize_input;
}

void
alsaio_stop(void)
{
    if (pcm_output != NULL) {
	snd_pcm_close(pcm_output);
    }
    if (pcm_input != NULL) {
	snd_pcm_close(pcm_input);
    }
}

#endif
