/*
 * (c) Copyright 1999, 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <math.h>
#include <inttypes.h>

#include "coeffs.h"

#define COEFFS_MAX_LINELEN 8192

static bool_t
parse_doublestr(double *dbl, char s[])
{
    char *p;
    uint64_t tmp, base;
    int n;
    
    if ((p = strstr(s, "0x")) != NULL) {
	/* raw format, 16 hex chars mapping directly to double raw data */
	p += 2;
	tmp = 0;
	base = 1;
	if (strlen(p) < 2 * sizeof(double)) {
	    return false;
	}
	for (n = 2 * sizeof(double) - 1; n >= 0 ; n--) {
	    switch (p[n]) {
	    case '0': break;
	    case '1': tmp += base; break;
	    case '2': tmp += 2 * base; break;
	    case '3': tmp += 3 * base; break;
	    case '4': tmp += 4 * base; break;
	    case '5': tmp += 5 * base; break;
	    case '6': tmp += 6 * base; break;
	    case '7': tmp += 7 * base; break;
	    case '8': tmp += 8 * base; break;
	    case '9': tmp += 9 * base; break;
	    case 'a': case 'A': tmp += 10 * base; break;
	    case 'b': case 'B': tmp += 11 * base; break;
	    case 'c': case 'C': tmp += 12 * base; break;
	    case 'd': case 'D': tmp += 13 * base; break;
	    case 'e': case 'E': tmp += 14 * base; break;
	    case 'f': case 'F': tmp += 15 * base; break;
	    default:
		return false;
	    }
	    base = base << 4;
	}
	memcpy(dbl, &tmp, sizeof(double));
    } else {
	/* decimal notation of a double */
	*dbl = strtod(s, &p);
	if (p == s) {
	    return false;
	}
    }
    return true;
}

static bool_t
isemptyline(char s[])
{
    int n;
    
    for (n = 0; s[n] != '\0' && s[n] != '\n'; n++) {
	if (s[n] != ' ' && s[n] != '\t') {
	    return false;
	}
    }
    return true;
}

#define COEFFS_PARSE_CLEANUP(str)                                              \
    free(s);                                                                   \
    for (n = 0; data[n] != NULL; n++) {                                        \
	free(data[n]);                                                         \
    }                                                                          \
    free(data);                                                                \
    fprintf(stderr, str);                                                      \
    return NULL;

#define COEFFS_COL_EXPAND 64

/*
 * Reads a stream containing double values in columns. The returned value is a
 * null terminated array of double arrays (columns). The arrays have been
 * allocated with malloc and must be free'd (automatically done with
 * coeffs_free) The double values can either be in human readable form (i.e.
 * 1.23432, -3.141, etc.) or in raw hex format, (i.e. 0x3F9FB51FEF9FA680). Both
 * forms can be used in the same file. If a value is missing in a column, 0 is
 * put in its place.
 */

double **
coeffs_parse(int *len, /* receives the length of the columns */
	     FILE *stream)
{
    char *s = malloc(COEFFS_MAX_LINELEN), *p;
    double **data = malloc(2 * sizeof(double *));
    void *pv;
    int n, i, n_cols = 1, cap = 0;

    if (len == NULL || stream == NULL) {
	free(s); free(data);
	fprintf(stderr, "coeffs_parse: at least one input parameter is "
		"invalid\n");	
	return NULL;
    }
    if (s == NULL || data == NULL) {
	free(s); free(data);
	fprintf(stderr, "coeffs_parse: could not allocate memory\n");
	return NULL;
    }
    
    s[COEFFS_MAX_LINELEN-1] = '\0';
    *len = 0;
    data[0] = NULL;
    data[1] = NULL;
    while (fgets(s, COEFFS_MAX_LINELEN-1, stream) != NULL && !isemptyline(s)) {
	s[strlen(s)-1] = '\0';
	p = s;
	for (p = s, n = 0; *p != '\0'; n++) {
	    if (*len == cap) {
		/* extend length of columns */
		cap += COEFFS_COL_EXPAND;
		for (i = 0; i < n_cols; i++) {
		    if ((pv = realloc(data[i], cap * sizeof(double)))
			== NULL)
		    {
			COEFFS_PARSE_CLEANUP("coeffs_parse: could not allocate"
					     " memory\n");
		    }
		    data[i] = pv;
		    bzero(&data[i][cap - COEFFS_COL_EXPAND],
			  COEFFS_COL_EXPAND * sizeof(double));
		}
	    }
	    if (n == n_cols) {
		/* expand one column */
		if ((pv = realloc(data, (++n_cols + 1) * sizeof(double *)))
		    == NULL)
		{
		    COEFFS_PARSE_CLEANUP("coeffs_parse: could not allocate "
					 "memory\n");
		}
		data = pv;
		data[n_cols] = NULL;
		if ((data[n] = malloc(cap * sizeof(double))) == NULL) {
		    COEFFS_PARSE_CLEANUP("coeffs_parse: could not allocate "
					 "memory\n");
		}
		bzero(data[n], cap * sizeof(double));
	    }
	    if (!parse_doublestr(&data[n][*len], p)) {
		for (n = 0; data[n] != NULL; n++) {
		    free(data[n]);
		}
		free(data);
		fprintf(stderr, "coeffs_parse: invalid string \"%s\" on line "
			"%d\n",	p, *len + 1);
		free(s);
		return NULL;
	    }
	    while (isspace(*p++));
	    while (!isspace(*p) && *p != '\0') p++;
	}
	*len += 1;
    }
    free(s);
    if (data[0] == NULL) {
	free(data);
	data = NULL;
    }
    return data;
}

#undef COEFFS_PARSE_CLEANUP

/*
 * Map floating point values to integers. If the input array is {-0.1, 0.1}
 * and bits for the mapping is set to 8, the output array will be {-127, 127},
 * and the scale factor 1270.
 */
int32_t *
coeffs_requantisise(double coeffs[], /* fp values to be mapped to integers */
		    int len,         /* length of coeffs */
		    int bits,        /* bits used for the mapping */
		    int32_t *multiplier, /* receives the scale factor */
		    double *error) /* (may be NULL) receives the largest
				    * requantisation error, i.e. 0.1 means that
				    * at least one requantisised integer value
				    * differs 10% from its floating point
				    * counterpart */
{
    int n, *icoeffs;
    double min, max;
    double a;

    if (coeffs == NULL || len < 1 || bits < 8 || bits > 32 ||
	multiplier == NULL)
    {
	fprintf(stderr, "coeffs_requantisise: at least one "
		"input parameter is invalid\n");
	return NULL;	    
    }
    if ((icoeffs = malloc(len * sizeof(int32_t))) == NULL) {
	fprintf(stderr, "coeffs_requantisise: could not allocate memory\n");
	return NULL;
    }
    
    /* get the range */
    min = max = coeffs[0];
    for (n = 1; n < len; n++) {
	if (coeffs[n] < min) {
	    min = coeffs[n];
	}
	if (coeffs[n] > max) {
	    max = coeffs[n];
	}
    }

    /* calculate multiplier */
    if (fabs(min) < fabs(max)) {
	a = fabs(max) / (double)((1 << (bits - 1)) - 1);	
    } else {
	a = fabs(min) / (double)((1 << (bits - 1)) - 1);
    }
    a = 1 / floor(1 / a);
    *multiplier = (int32_t)rint(1 / a);

    /* calculate integer coefficients */
    if (error != NULL) {
	*error = 0;
    }
    for (n = 0; n < len; n++) {
	icoeffs[n] = (int)rint(coeffs[n] / a);
	/*
	if (icoeffs[n] >= 0) {
	    fprintf(stderr, " %d\n", icoeffs[n]);
	} else {
	    fprintf(stderr, "%d\n", icoeffs[n]);
	}
	*/
	if (error != NULL) {
	    if (*error < fabs(coeffs[n] / ((double)icoeffs[n] * a)) - 1) {
		*error = fabs(coeffs[n] / ((double)icoeffs[n] * a)) - 1;
	    }
	}
    }
    
    return icoeffs;    
}

bool_t
coeffs_print(FILE *stream,
	     double *coeffs[],
	     int len,
	     bool_t raw)
{
    uint64_t tmp;
    int n, i;
    char s[50];
    
    if (stream == NULL || coeffs == NULL || len < 0) {
	fprintf(stderr, "coeffs_print: at least one input parameter is "
		"invalid\n");
	return false;
    }    
    
    for (n = 0; n < len; n++) {
	for (i = 0; coeffs[i] != NULL; i++) {
	    if (i != 0) {
		if (fprintf(stream, "\t") != 1) {
		    fprintf(stderr, "coeffs_print: print failed\n");
		    return false;
		}
	    }
	    if (raw) {
		memcpy(&tmp, &coeffs[i][n], sizeof(double));
		if (fprintf(stream, "0x%.16llX", tmp) != 18) {
		    fprintf(stderr, "coeffs_print: print failed\n");
		    return false;
		}
	    } else {
		if (coeffs[i][n] < 0) {
		    sprintf(s, "%.16e", coeffs[i][n]);
		} else {
		    sprintf(s, " %.16e", coeffs[i][n]);
		}
		if (fprintf(stream, "%s", s) != strlen(s)) {
		    fprintf(stderr, "coeffs_print: print failed\n");
		    return false;
		}
	    }
	}
	if (fprintf(stream, "\n") != 1) {
	    fprintf(stderr, "coeffs_print: print failed\n");
	    return false;
	}
    }
    
    return true;
}

void
coeffs_free(double *coeffs[])
{
    int n;

    for (n = 0; coeffs[n] != NULL; n++) {
	free(coeffs[n]);
    }
    free(coeffs);
}

/* parse a string containing a fraction (x/y), or a plain number (x) */
double
coeffs_fractionparse(char *s)
{
    char *p;

    if ((p = strchr(s, '/')) == NULL) {
	return atof(s);
    }
    *p = '\0';
    return atof(s) / atof(++p);
}

