/*
 * (c) Copyright 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <inttypes.h>
#include <math.h>

#include <rfftw.h>

#include "defs.h"
#include "coeffs.h"
#include "timestamp.h"

static rfftw_plan fftplan;
static rfftw_plan ifftplan;
static int nfft, nfft2;

static int
getpow2(int x)
{
    int n;
    
    if (x < 1) {
	return -1;
    }
    for (n = 0; (x & 1) == 0 && n < 32; x = x >> 1, n++);
    if (n == 32 || (x & ~1) != 0) {
	return -1;
    }
    return n;
}

static float *
make_coeffs(char filename[],
	    float attenuation)
{
    FILE *stream;
    double **input;
    float *coeffs, *dcoeffs;
    int n, len;
    
    if ((stream = fopen(filename, "rt")) == NULL) {
	fprintf(stderr, "Could not open file \"%s\" for reading\n", filename);
	return NULL;
    }
    if ((input = coeffs_parse(&len, stream)) == NULL) {
	fprintf(stderr, "Failed to parse coefficients\n");
	return NULL;
    }
    if (len != nfft2) {
	fprintf(stderr, "Warning: filter length is not %d, resizing\n", nfft2);
    }
    if ((dcoeffs = malloc(nfft * sizeof(float))) == NULL) {
	fprintf(stderr, "memory allocation failure\n");
	return NULL;
    }
    memset(dcoeffs, 0, nfft * sizeof(float));
    if (len > nfft2) {
	len = nfft2;
    }
    for (n = 0; n < len; n++) {
	dcoeffs[nfft2+n] = (float)(input[0][n] * attenuation);
    }
    if ((coeffs = malloc(nfft * sizeof(float))) == NULL) {
	fprintf(stderr, "memory allocation failure\n");
	free(dcoeffs);
	coeffs_free(input);
	return NULL;
    }
    rfftw_one(fftplan, dcoeffs, coeffs);
    for (n = 0; n < nfft; n++) {
	coeffs[n] /= (float)nfft;
    }
    return coeffs;
}

static void
fastconv(float *coeffs,
	 float *block)
{
    float bl[nfft], a;
    int n;

    rfftw_one(fftplan, block, bl);
    bl[0] *= coeffs[0];
    for (n = 1; n < nfft2; n++) {
	a = bl[n];
	bl[n] = a * coeffs[n] - bl[nfft-n] * coeffs[nfft-n];
	bl[nfft-n] = a * coeffs[nfft-n] + bl[nfft-n] * coeffs[n];
    }
    bl[nfft2] *= coeffs[nfft2];
    rfftw_one(ifftplan, bl, block);
}

static void
toconvblock(float *ch1block1,
	    float *ch2block1,
	    float *ch1block2,
	    float *ch2block2,
	    void *inbuf)
{
    int n;

    for (n = 0; n < nfft; n += 2) {
	ch1block2[n>>1] = (float)((int16_t *)inbuf)[n];
	ch2block2[n>>1] = (float)((int16_t *)inbuf)[n+1];
    }
    memcpy(&ch1block1[nfft2], ch1block2, nfft2 * sizeof(float));
    memcpy(&ch2block1[nfft2], ch2block2, nfft2 * sizeof(float));
}

static void
fromconvblock(float *ch1block,
	      float *ch2block,
	      void *outbuf)
{
    int n;

    for (n = 0; n < nfft; n += 2) {
	((int16_t *)outbuf)[n] = (int16_t)ch1block[n>>1];
	((int16_t *)outbuf)[n+1] = (int16_t)ch2block[n>>1];	
    }    
}

int
main(int argc,
     char *argv[])
{
    float *ch1_coeffs, *ch2_coeffs, *ch1_block[2], *ch2_block[2], att;
    void *inbuf, *outbuf;
    char *coeffs1, *coeffs2;
    int method;

    if (sizeof(fftw_real) != sizeof(float)) {
	fprintf(stderr, "ERROR: fftw was not compiled with --enable-float\n");
	return 1;
    }
    
    if (argc != 5 && argc != 6) {
	fprintf(stderr, "Usage: %s [-m] <ch1 coeffs> <ch2 coeffs> "
		"<attenuation in dB> <filter length>\n", argv[0]);
	return 1;
    }
    if (argc == 6) {
	if (strcmp(argv[1], "-m") != 0) {
	    fprintf(stderr, "unknown option \"%s\"\n", argv[1]);
	    return 1;
	}
	method = FFTW_MEASURE;
	coeffs1 = argv[2];
	coeffs2 = argv[3];
	att = atof(argv[4]);
	nfft = atoi(argv[5]);
    } else {
	method = FFTW_ESTIMATE;
	coeffs1 = argv[1];
	coeffs2 = argv[2];
	att = atof(argv[3]);
	nfft = atoi(argv[4]);
    }
    att = pow(10, -att / 20);
    if (getpow2(nfft) == -1) {
	fprintf(stderr, "filter length not power of two\n");
	return 1;
    }
    nfft2 = nfft;
    nfft *= 2;    
    
    fprintf(stderr, "generating fftw plans...");
    fftplan = rfftw_create_plan(nfft, FFTW_REAL_TO_COMPLEX, method);
    ifftplan = rfftw_create_plan(nfft, FFTW_COMPLEX_TO_REAL, method);
    fprintf(stderr, "finished\n");
    
    if ((ch1_coeffs = make_coeffs(coeffs1, att)) == NULL ||
	(ch2_coeffs = make_coeffs(coeffs2, att)) == NULL)
    {
	return 1;
    }
    
    inbuf = malloc(nfft2 * 4);
    outbuf = malloc(nfft2 * 4);
    ch1_block[0] = malloc(2*nfft * sizeof(float));
    ch2_block[0] = malloc(2*nfft * sizeof(float));
    ch1_block[1] = malloc(2*nfft * sizeof(float));
    ch2_block[1] = malloc(2*nfft * sizeof(float));
    memset(ch1_block[0], 0, 2*nfft * sizeof(float));
    memset(ch2_block[0], 0, 2*nfft * sizeof(float));
    memset(ch1_block[1], 0, 2*nfft * sizeof(float));
    memset(ch2_block[1], 0, 2*nfft * sizeof(float));

    fprintf(stderr, "starting to process incoming data\n");
    
    while (true) {
	if (fread(inbuf, nfft2 * 4, 1, stdin) != 1) break;
	toconvblock(ch1_block[0], ch2_block[0],
		    ch1_block[1], ch2_block[1], inbuf);
	fastconv(ch1_coeffs, ch1_block[0]);
	fastconv(ch2_coeffs, ch2_block[0]);
	fromconvblock(ch1_block[0], ch2_block[0], outbuf);
	if (fwrite(outbuf, nfft2 * 4, 1, stdout) != 1) break;

	if (fread(inbuf, nfft2 * 4, 1, stdin) != 1) break;
	toconvblock(ch1_block[1], ch2_block[1],
		    ch1_block[0], ch2_block[0], inbuf);
	fastconv(ch1_coeffs, ch1_block[1]);
	fastconv(ch2_coeffs, ch2_block[1]);
	fromconvblock(ch1_block[1], ch2_block[1], outbuf);
	if (fwrite(outbuf, nfft2 * 4, 1, stdout) != 1) break;
    }

    fprintf(stderr, "finished\n");
    return 0;
}
