/*
 * (c) Copyright 1999, 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#ifndef FILTER_H_
#define FILTER_H_

#include <inttypes.h>

#include "defs.h"
#include "sample.h"

#define MAX_CHANNELS 100

struct filter_t_;
typedef struct filter_t_ filter_t;

struct io_func {
    int (*fillbuf)(void *);
    void (*flushbuf)(void *, int);
    void (*stop_io)(void);    
};

/*
 * This struct contains information about a filter processor, that is a
 * subsystem that filters samples. When a filter processor is initialised it
 * should return an initialised filterproc_t that can be passed to filter_new.
 *
 * The fields:
 *
 * private    -- the filter's private data, which is given as first parameter
 *               to the 'process_sample' and 'toggle_processing' functions.
 * multiplier -- how many times the filter scales the sample range.
 * db_gain    -- how many decibels a processed sample should be scaled in
 *               filter.c.
 * process_sample -- this function should process a sample, which is a 32 bit
 *                   signed integer, range -2147483648 to 2147483647. The return
 *                   sample is a 64 bit integer, and should have the
 *                   corresponding range scaled with 'multiplier'
 * toggle_processing -- toggle filter processing on and off. 'process_sample'
 *                      will still be called so the filter processing can
 *                      maintain its internal state.
 */
typedef struct filterproc_t_ {
    void *private;      
    int32_t multiplier;
    double db_gain;
    int64_t (*process_sample)(void *private, int32_t sample);
    void (*toggle_processing)(void *private);
} filterproc_t;

filter_t *
filter_new(int fragsize,
	   int n_frags,
	   int n_channels,
	   struct sample_format *insf,
	   struct sample_format *outsf,
	   filterproc_t *filterproc[],
	   struct io_func *io_func);

void
filter_run(filter_t *filter,
	   bool_t show_progress);

void
filter_toggle_processing(filter_t *filter);

void
filter_delete(filter_t *filter);

#endif
