/*
 * (c) Copyright 1999, 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU Public License as published by the Free Software Foundation.
 *
 */
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <inttypes.h>
#include <gsl/gsl_fft_real.h>
#include <gsl/gsl_fft_halfcomplex.h>

#include "toolbox.h"
#include "coeffs.h"
#include "spline.h"
#include "timestamp.h"

#define PRESENTATION_STRING \
"
F I R D -- a FIR filter designer                           Experimental version
================================                              (c) Anders Torger
                                                                   October 2000

"

#define USAGE_STRING \
"Usage: %s [-dhi] [-g <gain in dB>] [-r <precision>] [-s <sample frequency>]
        <filter design algorithm> <fda options>

 -d phase values are in degrees
 -h apply hamming window on output. Only suitable for symmetrical filters.
 -i magnitude values are in decibel
 -g <gain in dB> filter gain in decibel (may be negative). Default: 0.0
 -r <precision> write raw output in either SINGLE or DOUBLE precision.
 -s <sample frequency> in Hz, default: 44100
 <number of taps> the length of the produced filter. If set to zero or negative,
    the filter algorithm will itself choose a suiting length.

 Filter design algorithms and their options:
 LOWPASS -- lowpass filter
   <cutoff frequency>
 HIGHPASS -- highpass filter
   <cutoff frequency>
 BANDPASS -- bandpass filter
   <low cutoff frequency> <high cutoff frequency>
 BANDSTOP -- bandstop filter
   <low cutoff frequency> <high cutoff frequency>
 INVERT -- mixed phase inversion
   <delay in samples> [<impulse response filename>] (default: stdin)
 MINVERT -- minimum phase inversion
   [<impulse response filename>] (default: stdin)
 LINEARPHASE -- make filter from power response, with linear phase
   [<power response filename>] (default: stdin)
 MINPHASE -- make filter from power response, with mininum phase
   [<power response filename>] (default: stdin)
 ALLPASS -- make an allpass filter from phase response
   [<phase response filename>] (default: stdin)
 CONVOLVE -- combine two filters into one (warning: slow for long filters)
   [<impulse response 1 filename> <impulse response 2 filename> |
    <imp1/imp2 filename>] (default: imp1/imp2 from stdin)
"

#define FILTER_LOWPASS 1
#define FILTER_HIGHPASS 2
#define FILTER_BANDPASS 3
#define FILTER_BANDSTOP 4
#define FILTER_INVERT 5
#define FILTER_MINVERT 6
#define FILTER_LINEARPHASE 7
#define FILTER_MINPHASE 8
#define FILTER_ALLPASS 9
#define FILTER_CONVOLVE 10


static double *
hamming_window(int len)
{
    double *win;
    int n, i;
    
    if ((win = malloc(len * sizeof(double))) == NULL) {
	fprintf(stderr, "hamming_window: memory allocation failed\n");
	return NULL;
    }
    i = len / 2;
    if (len % 2 != 0) {
	for (n = 0; n <= i; n++) {
	    win[n+i] = 0.54 + 0.46 * cos(2 * M_PI * n / (len-1));
	}
    } else {
	for (n = 0; n < len / 2; n++) {
	    win[n+i] = 0.54 + 0.46 * cos(2 * M_PI * (2*n+1) / (2*(len-1)));
	}
    }
    for(n = 0; n < i; n++) {
	win[n] = win[len-1-n];
    }
    return win;
}

#define FILTER_RESPONSE(normal, zero) {                                        \
    fpos = &filter[(n_taps - 1) >> 1];                                         \
    for (n = -((n_taps - 1) >> 1); n != 0; n++) {                              \
	fpos[n] = normal;                                                      \
    }                                                                          \
    fpos[0] = zero;                                                            \
    for (n = 1; n <= n_taps >> 1; n++) {                                       \
	fpos[n] = normal;                                                      \
    }                                                                          \
}

static double *
pass_filter(double low_freq,
	    double high_freq,
	    double sampling_freq,	    
	    int n_taps,
	    int type)
{
    double *filter, *fpos;
    int n;

    if (type == FILTER_LOWPASS && high_freq != -1.0) {
	fprintf(stderr, "pass_filter: high_freq is not used to create "
		"a lowpass filter, set it to -1.0\n");
	return NULL;
    }
    if (type == FILTER_HIGHPASS && low_freq != -1.0) {
	fprintf(stderr, "pass_filter: low_freq is not used to create "
		"a highpass filter, set it to -1.0\n");
	return NULL;
    }
    
    if ((filter = malloc(n_taps * sizeof(double))) == NULL) {
	fprintf(stderr, "pass_filter: memory allocation failed\n");
	return NULL;
    }
    
    low_freq = 2 * low_freq / sampling_freq * M_PI;
    high_freq = 2 * high_freq / sampling_freq * M_PI;
    
    switch (type) {

    case FILTER_LOWPASS:
	FILTER_RESPONSE(sin((double)n * low_freq) / ((double)n * M_PI),
			low_freq / M_PI);
	break;
	
    case FILTER_HIGHPASS:
	FILTER_RESPONSE(-sin((double)n * high_freq) / ((double)n * M_PI),
			1.0 - high_freq / M_PI);
	break;
	
    case FILTER_BANDPASS:
	FILTER_RESPONSE((sin((double)n * high_freq) -
			 sin((double)n * low_freq)) / ((double)n * M_PI),
			(high_freq - low_freq) / M_PI);
	break;
	
    case FILTER_BANDSTOP:
	FILTER_RESPONSE((sin((double)n * low_freq) -
			 sin((double)n * high_freq)) / ((double)n * M_PI),
			1.0 + (high_freq - low_freq) / M_PI);
	break;
	
    default:
	fprintf(stderr, "pass_filter: invalid type (%d)\n", type);
	free(filter);
	return NULL;
    }

    return filter;
}

#undef FILTER_RESPONSE

static bool_t
phase_decomposition(double **ep,
		    double **mp,
		    double imp[],
		    int len)
{
    int n;
    double *tmp;
    gsl_complex_packed_array complex;
    
    complex = malloc(len * sizeof(gsl_complex));
    *ep = malloc(len * sizeof(double));
    *mp = malloc(len * sizeof(double));
    tmp = malloc(len * sizeof(double));
    
    if (complex == NULL || *ep == NULL || *mp == NULL || tmp == NULL) {
	fprintf(stderr, "phase_decomposition: memory allocation failed\n");
	free(complex); free(*ep); free(*mp); free(tmp);
	return false;
    }
    
    fprintf(stderr, "separating minumum and excess phase components...");
    for (n = 0; n < len; n++) {
	complex[2*n] = imp[n];
	complex[2*n+1] = 0;
    }
    if (!tb_real_cepstrum(tmp, *ep, *mp, complex, len)) {
	fprintf(stderr, "phase_decomposition: cepstrum calculation failed\n");
	free(complex); free(*ep); free(*mp); free(tmp);
	return false;
    }
    free(complex);
    free(tmp);
    fprintf(stderr, "finished!\n");
    
    return true;
}

static int
log2newlen(int len,
	   int taps)
{
    uint32_t n, i;
    int xlen, order = 0, newlen;
    
    if (len < 2 * taps) {
	xlen = 2 * taps;
    } else {
	xlen = len;
    }
    if (tb_getlog2(xlen) == -1) {
	for (n = 0, i = xlen; n < 32; i >>= 1, n++) {
	    if ((i & 1) != 0) {
		order = n;
	    }
	}
	order++;
    } else {
	order = tb_getlog2(xlen);	
    }
    newlen = 1 << order;    
    if (newlen < 512) {
	newlen = 512;
    }
    return newlen;
}

static double *
minimum_phase_filter(double freq[],
		     double mag[],
		     int len,
		     int taps,
		     double sfreq)
{
    double *mags, *out, *mp, *ep;
    int newlen, n;
    
    newlen = log2newlen(len, taps);
    if ((mags = tb_interpolate(NULL, freq, mag, len, newlen, true, 0,
			       sfreq / 2.0, 1.0, 1.0)) == NULL)
    {
	fprintf(stderr, "minimum_phase_filter: interpolation failed\n");
	return NULL;
    }
    if ((out = malloc(2 * newlen * sizeof(double))) == NULL) {
	fprintf(stderr, "minimum_phase_filter: memory allocation failure\n");
	free(mags);
	return NULL;
    }

    out[0] = mags[0];
    for (n = 1; n < newlen; n++) {
	out[n] = mags[n];
	out[(newlen<<1)-n] = 0;
    }
    out[newlen] = mags[newlen-1];
    free(mags);    
    gsl_fft_halfcomplex_radix2_inverse(out, 1, 2 * newlen);
    if (!phase_decomposition(&ep, &mp, out, 2 * newlen)) {
	fprintf(stderr, "minimum_phase_filter: phase decomoposition failed\n");
	free(out);
	return NULL;
    }
    free(ep);
    free(out);
    return mp;
}

static double *
linear_phase_filter(double freq[],
		    double mag[],
		    int len,
		    int taps,
		    double sfreq)
{
    double *out, *mags, rad;
    int n, newlen;

    newlen = log2newlen(len, taps);
    if ((mags = tb_interpolate(NULL, freq, mag, len, newlen, true, 0,
			       sfreq / 2.0, 1.0, 1.0)) == NULL)
    {
	fprintf(stderr, "linear_phase_filter: interpolation failed\n");
	return NULL;
    }    
    if ((out = malloc(2 * newlen * sizeof(double))) == NULL) {
	fprintf(stderr, "linear_phase_filter: memory allocation failure\n");
	free(mags);
	return NULL;
    }

    out[0] = mags[0];
    for (n = 1; n < newlen; n++) {
	rad = -0.5 * (double)taps * M_PI * (double)n / (double)newlen;
	out[n] = cos(rad) * mags[n];
	out[(newlen<<1)-n] = sin(rad) * mags[n];
    }
    out[newlen] = mags[newlen-1];
    free(mags);
    
    gsl_fft_halfcomplex_radix2_inverse(out, 1, 2 * newlen);
    return out;
}

static double *
allpass_filter(double freq[],
	       double phase[],
	       int len,
	       int taps,
	       double sfreq)
{
    double *out, *phases, rad;
    int n, newlen;

    newlen = log2newlen(len, taps);
    if ((phases = tb_interpolate(NULL, freq, phase, len, newlen, true, 0,
				 sfreq / 2.0, 0.0, 0.0)) == NULL)
    {
	fprintf(stderr, "allpass_filter: interpolation failed\n");
	return NULL;
    }    
    if ((out = malloc(2 * newlen * sizeof(double))) == NULL) {
	fprintf(stderr, "allpass_filter: memory allocation failure\n");
	free(phases);
	return NULL;
    }

    out[0] = 0;
    for (n = 1; n < newlen; n++) {
	rad = -0.5 * (double)taps * M_PI * (double)n / (double)newlen;
	out[n] = cos(rad + phases[n]);
	out[(newlen<<1)-n] = sin(rad + phases[n]);
    }
    out[newlen] = 0;
    free(phases);
    
    gsl_fft_halfcomplex_radix2_inverse(out, 1, 2 * newlen);
    return out;    
}

static void
convolve(double out[],
	 double imp[],
	 int len,
	 double coeffs[],
	 int order)
{
    int n, i;

    for (i = 0; i < len; i++) {
	for (out[i] = 0, n = 0; n < order; n++) {
	    if (i - n >= 0) {
		out[i] += imp[i-n] * coeffs[n];
	    }
	}
    }
}

static double *
excess_phase_inverse(double epimp[],
		     int len,
		     int order,
		     int delay)
{
    double *a;
    int n;
    double *epf, *tmp;

    if ((tmp = malloc(order * sizeof(double))) == NULL) {
	fprintf(stderr, "excess_phase_inverse: memory allocation failed\n");
	return NULL;
    }
    
    fprintf(stderr, "inverting excess phase...");
    for (n = 0; n < delay; n++) {
	tmp[n] = epimp[delay - n];
    }
    bzero(tmp + delay, (order - delay) * sizeof(double));
    
    a = tb_cross_correlation(epimp, len);
    epf = tb_levinson(&a[len - 1], len, tmp, order);
    free(a);
    free(tmp);
    fprintf(stderr, "finished!\n");

    return epf;    
}

static double *
minimum_phase_inverse(double mpimp[],
		      int len,
		      int order)
{
    double *a, *mpf;
    
    fprintf(stderr, "inverting minimum phase...");
    a = tb_cross_correlation(mpimp, len);
    mpf = tb_levinson_durbin(&a[len - 1], len, order);
    free(a);
    fprintf(stderr, "finished!\n");

    return mpf;
}

static double *
mixed_phase_inverse(double imp[],
		    int len,
		    int order,
		    int delay)
{

    double *epf, *mpf, *mp, *ep, *filter;
    
    if (!phase_decomposition(&ep, &mp, imp, len)) {
	fprintf(stderr, "mixed_phase_inverse: phase decomposition failed\n");
	return NULL;
    }

    if ((epf = excess_phase_inverse(ep, len, order, delay)) == NULL) {
	fprintf(stderr, "mixed_phase_inverse: inversion failed\n");
	free(ep); free(mp);
	return NULL;
    }
    free(ep);
    
    if ((mpf = minimum_phase_inverse(mp, len, order)) == NULL) {
	fprintf(stderr, "mixed_phase_inverse: inversion failed\n");
	free(epf); free(mp);
	return NULL;
    }
    free(mp);
    
    if ((filter = malloc(order * sizeof(double))) == NULL) {
	fprintf(stderr, "mixed_phase_inverse: memory allocation failed\n");
	free(epf); free(mpf);
	return NULL;	
    }
        
    fprintf(stderr, "convolving components into complete inverse...");
    convolve(filter, mpf, order, epf, order);
    free(epf);
    free(mpf);
    fprintf(stderr, "finished!\n");
    
    return filter;
}

/*
 * Set of experimental functions which is not yet used. Their purpose is
 * to help psycho-acoustic FIR filter design.
 */

/*
static double *
sliding_lowpass(double in[],
		int len,
		double times[],
		double freqs[],
		int tflen,
		int n_taps,
		double sfreq)
{
    uint64_t t1, t2;
    double *curfreq = NULL, *curtime = NULL, *out = NULL;
    double *fqs = NULL, *fhalf = NULL, *ntab = NULL, lastfreq, maxfreq, a;
    int n, i, k, htaps;
    
    timestamp(&t1);

    if (times[0] != 0 || times[tflen-1] >= 0) {
	fprintf(stderr, "sliding_lowpass: times array must start at time 0 "
		"and be open ended (last element < 0)\n");
	return NULL;
    }
    
    htaps = n_taps>>1;
    if ((fhalf = malloc((htaps+1) * sizeof(double))) == NULL ||
	(ntab = malloc((htaps+1) * sizeof(double))) == NULL ||
	(curtime = malloc(len * sizeof(double))) == NULL ||
	(curfreq = malloc(len * sizeof(double))) == NULL ||
	(fqs = malloc(tflen * sizeof(double))) == NULL ||
	(out = malloc(len * sizeof(double))) == NULL)
    {
	fprintf(stderr, "sliding_lowpass: memory allocation failed\n");
	free(fhalf); free(ntab); free(curtime); free(curfreq); free(fqs);
	free(out);
	return NULL;
    }
    maxfreq = sfreq / sfreq * M_PI;
    for (n = 0; n < tflen; n++) {
	if (freqs[n] < 0) {
	    freqs[n] = sfreq / 2;
	}
	fqs[n] = 2 * freqs[n] / sfreq * M_PI;
    }    
    for (n = 0, a = 1.0 / sfreq; n < len; n++) {
	curtime[n] = (double)n * a;
    }
    times[tflen-1] = curtime[n-1];
    linear(curfreq, curtime, times, fqs, tflen, len);
    free(curtime);
    free(fqs);

    for (n = 1; n <= htaps; n++) {
	ntab[n] = 1.0 / M_PI / (double)n;
    }

    lastfreq = -1;
    for (i = 0; i < len; i++) {
	if (curfreq[i] != lastfreq && curfreq[i] < maxfreq) {
	    for (n = 1; n <= htaps; n++) {
		fhalf[n] = sin((double)n * curfreq[i]) * ntab[n];
	    }
	    fhalf[0] = curfreq[i] / M_PI;
	}
	if (curfreq[i] < maxfreq) {
	    out[i] = fhalf[0] * in[i];	    
	    for (n = htaps; n > 0; n--) {
		if ((k = i + n) >= 0 && k < len) {
		    out[i] += in[k] * fhalf[n];
		}
	    }
	    for (n = 1; n <= htaps; n++) {
		if ((k = i - n) >= 0 && k < len) {
		    out[i] += in[k] * fhalf[n];
		}
	    }
	} else {
	    out[i] = in[i];
	}
	lastfreq = curfreq[i];
    }
    free(curfreq);
    free(fhalf);
    free(ntab);
    
    timestamp(&t2);
    fprintf(stderr, "%f\n", (double)(t2 - t1) / 266000.0);

    return out;
}

static void
flatten(double imp[],
	int len,
	double freq,
	double sfreq)
{
    int n, i;
    double a;
    
    gsl_fft_real_radix2_transform(imp, 1, len);

    a = 0;
    for (i = n = freq * 0.75 / sfreq * len; n < freq / sfreq * len; n++) {
	a += sqrt(imp[n] * imp[n] + imp[len - n] * imp[len - n]);	
    }
    a /= (n - i);
    
    for (; n < len / 2; n++) {
	imp[n] = a;
	imp[len - n] = 0;
    }
    
    gsl_fft_halfcomplex_radix2_inverse(imp, 1, len);    
}

static void
flatten_magresp(double imp[],
		int len,
		double freq,
		double sfreq)
{
    int n, i;
    double a, f;
    
    gsl_fft_real_radix2_transform(imp, 1, len);

    a = 0;
    for (i = n = freq * 0.75 / sfreq * len; n < freq / sfreq * len; n++) {
	a += sqrt(imp[n] * imp[n] + imp[len - n] * imp[len - n]);	
    }
    a /= (n - i);
    
    for (; n < len / 2; n++) {
	f = a / sqrt(imp[n] * imp[n] + imp[len - n] * imp[len - n]);
	imp[n] *= f;
	imp[len - n] *= f;
    }
    
    gsl_fft_halfcomplex_radix2_inverse(imp, 1, len);    
}


static void
apply_magresp(double imp[],
	      double mag[],
	      int len)
{
    int n;
    double f;
    
    gsl_fft_real_radix2_transform(imp, 1, len);
    
    f = 0;
    for (n = 1; n < len / 2; n++) {
	f = mag[n] / sqrt(imp[n] * imp[n] + imp[len - n] * imp[len - n]);
	imp[n] *= f;
	imp[len - n] *= f;	
    }
    imp[len/2] *= f;
    
    gsl_fft_halfcomplex_radix2_inverse(imp, 1, len);
}

static double *
get_smoothed_magresp(double imp[],
		     int len,
		     double fbands[],
		     int fblen,
		     double sfreq)
{
    int n, i = 0, k = 0;
    double *x, *y, *a, *b, *im, fb, f;

    im = malloc(len * sizeof(double));
    memcpy(im, imp, len  * sizeof(double));
    x = malloc(len * sizeof(double));
    y = malloc(len * sizeof(double));
    a = malloc(len * sizeof(double));
    b = malloc(len * sizeof(double));
    
    gsl_fft_real_radix2_transform(im, 1, len);

    fb = 0;
    for (i = 0, n = 1; i < fblen - 1 && n < len / 2; i++) {
	k = 0;
	y[i] = 0;
	f = (double)n / (double)len * sfreq;
	do {
	    y[i] += sqrt(im[n] * im[n] + im[len - n] * im[len - n]);
	    b[n] = (double)n / (double)len * sfreq;
	    n++;
	    k++;
	} while (b[n-1] < fbands[i+1] && n < len / 2);
	x[i] = b[n-1] + (f - b[n-1]) / 2;
	y[i] /= k;
	fprintf(stderr, "%f %f %f %f\n", fbands[i], fbands[i+1],
		fbands[i+1] - fbands[i], 20 * log10(y[i]));
    }

    x[0] = 0;
    x[i] = b[n-1];
    y[i] = y[i-1];
    i++;
    linear(a, b, x, y, i, n);
    
    return a;
}

static inline double
log2(double x)
{
    return log(x) / log(2);
}

static void
slope(double imp[],
      int len,
      double freq,
      double db_per_octave,
      double sfreq)
{
    int n;
    double a = 0, db, cfreq;
    
    gsl_fft_real_radix2_transform(imp, 1, len);

    for (db = 0, n = freq / sfreq * len; n < len / 2; n++) {
	cfreq = (double)n / (double)len * sfreq;
	db = db_per_octave * log2(cfreq / freq);
	a = pow(10, db / 20);
	imp[n] *= a;
	imp[len - n] *= a;
    }
    imp[len/2] *= a;
    gsl_fft_halfcomplex_radix2_inverse(imp, 1, len);    
}
*/

int
main(int argc,
     char *argv[])
{
    double **input, **input2, *out = NULL, *imp, *imp2, *phase, *mag, *win;
    double sfreq = 44100, gain = 0.0, c1, c2, *freq;
    bool_t dbinput = false, degreeinput = false, rawoutput = false;
    bool_t rawsingle = false, apply_hamming = false;
    int n, len, len2, taps, delay = 0, fdes;
    float *f = NULL;
    FILE *stream = stdin;
    size_t outsize;

    fprintf(stderr, PRESENTATION_STRING);

    if (argc < 2) {
	fprintf(stderr, USAGE_STRING, argv[0]);
	return 1;
    }
    for (n = 1; n < argc - 2 && argv[n][0] == '-'; n++) {
	switch (argv[n][1]) {
	case 'd':
	    degreeinput = true;
	    break;
	case 'h':
	    apply_hamming = true;
	    break;
	case 'i':
	    dbinput = true;
	    break;
	case 'g':
	    gain = atof(argv[++n]);
	    break;
	case 'r':
	    n++;
	    rawoutput = true;
	    if (strcasecmp(argv[n], "SINGLE") == 0) {
		rawsingle = true;
	    } else if (strcasecmp(argv[n], "DOUBLE") == 0) {
		rawsingle = false;
	    } else {
		fprintf(stderr, "Unkown precision \"%s\", choose either SINGLE "
			"or DOUBLE\n", argv[n]);
		return 1;
	    }
	case 's':
	    if ((sfreq = atof(argv[++n])) <= 0) {
		fprintf(stderr, "Invalid sample frequency: \"%s\"\n", argv[n]);
		return 1;
	    }
	    break;
	default:
	    fprintf(stderr, USAGE_STRING, argv[0]);
	    return 1;
	}
    }
    taps = atoi(argv[n]);
    if (++n == argc) {
	fprintf(stderr, "Missing filter design algorithm\n");
	return 1;
    }
    if (strcasecmp(argv[n], "LOWPASS") == 0) {
	fdes = FILTER_LOWPASS;
    } else if (strcasecmp(argv[n], "HIGHPASS") == 0) {
	fdes = FILTER_HIGHPASS;
    } else if (strcasecmp(argv[n], "BANDPASS") == 0) {
	fdes = FILTER_BANDPASS;
    } else if (strcasecmp(argv[n], "BANDSTOP") == 0) {
	fdes = FILTER_BANDSTOP;
    } else if (strcasecmp(argv[n], "INVERT") == 0) {
	fdes = FILTER_INVERT;
    } else if (strcasecmp(argv[n], "MINVERT") == 0) {
	fdes = FILTER_MINVERT;
    } else if (strcasecmp(argv[n], "LINEARPHASE") == 0) {
	fdes = FILTER_LINEARPHASE;
    } else if (strcasecmp(argv[n], "MINPHASE") == 0) {
	fdes = FILTER_MINPHASE;
    } else if (strcasecmp(argv[n], "ALLPASS") == 0) {
	fdes = FILTER_ALLPASS;
    } else if (strcasecmp(argv[n], "CONVOLVE") == 0) {
	fdes = FILTER_CONVOLVE;
    } else {
	fprintf(stderr, "Unknown filter design algorithm \"%s\"\n", argv[n]);
	return 1;
    }

    switch (fdes) {
    case FILTER_LOWPASS:
    case FILTER_HIGHPASS:
    case FILTER_BANDPASS:
    case FILTER_BANDSTOP:
	if (n == argc - 1) {
	    fprintf(stderr, "Missing cutoff frequency parameter\n");
	    return 1;
	}
	if ((c1 = atof(argv[++n])) <= 0) {
	    fprintf(stderr, "Invalid cutoff frequency: \"%s\"\n", argv[n]);
	    return 1;
	}
	
	if (fdes == FILTER_LOWPASS) {
	    out = pass_filter(c1, -1.0, sfreq, (taps <= 0) ? 1024 : taps,
			      FILTER_LOWPASS);
	    break;
	} else if (fdes == FILTER_HIGHPASS) {
	    out = pass_filter(-1.0, c1, sfreq, (taps <= 0) ? 1024 : taps,
			      FILTER_HIGHPASS);
	    break;
	}
       
	if (n == argc - 1) {
	    fprintf(stderr, "Missing cutoff frequency parameter\n");
	    return 1;
	}
	if ((c2 = atof(argv[++n])) <= 0) {
	    fprintf(stderr, "Invalid cutoff frequency: \"%s\"\n", argv[n]);
	    return 1;
	}
	if (fdes == FILTER_BANDPASS) {
	    out = pass_filter(c1, c2, sfreq, (taps <= 0) ? 1024 : taps,
			      FILTER_BANDPASS);
	} else {
	    out = pass_filter(c1, c2, sfreq, (taps <= 0) ? 1024 : taps,
			      FILTER_BANDSTOP);
	}
	break;
	

    case FILTER_MINVERT:
    case FILTER_INVERT:
	if (fdes == FILTER_INVERT) {
	    if (n == argc - 1) {
		fprintf(stderr, "Missing delay parameter\n");
		return 1;
	    }
	    if ((delay = atoi(argv[++n])) < 0) {
		fprintf(stderr, "Invalid delay: \"%s\"\n", argv[n]);
	    }
	}
	if (n < argc - 1) {
	    if ((stream = fopen(argv[++n], "rt")) == NULL) {
		fprintf(stderr, "Could not open file \"%s\" for reading: %s\n",
			argv[n], strerror(errno));
		return 1;
	    }
	}
	if ((input = coeffs_parse(&len, stream)) == NULL) {
	    fprintf(stderr, "Failed to parse impulse response\n");
	    return 1;
	}
	fclose(stream);
	if (input[1] != NULL) {
	    fprintf(stderr, "Expected only one column in input\n");
	    return 1;
	}
	if (taps <= 0) {
	    taps = len;
	} else if (taps > len) {
	    fprintf(stderr, "Filter length cannot be longer than input\n");
	    return 1;
	}
	if ((imp = tb_log2lengthen(&len, input[0])) == NULL) {
	    fprintf(stderr, "Failed to extend impulse response");
	    return 1;
	}
	coeffs_free(input);
	if (len == taps) {
	    if ((imp = realloc(imp, 2 * len * sizeof(double))) == NULL) {
		fprintf(stderr, "Memory allocation failed\n");
		return 1;
	    }
	    memset(&imp[len], 0, len * sizeof(double));
	    len *= 2;
	}
	if (fdes == FILTER_INVERT) {
	    out = mixed_phase_inverse(imp, len, taps, delay);
	} else {
	    out = minimum_phase_inverse(imp, len, taps);
	}
	break;


    case FILTER_LINEARPHASE:
    case FILTER_MINPHASE:
	if (n < argc - 1) {
	    if ((stream = fopen(argv[++n], "rt")) == NULL) {
		fprintf(stderr, "Could not open file \"%s\" for reading: %s\n",
			argv[n], strerror(errno));
		return 1;
	    }
	}
	if ((input = coeffs_parse(&len, stream)) == NULL) {
	    fprintf(stderr, "Failed to parse power response\n");
	    return 1;
	}
	fclose(stream);
	if (input[1] == NULL || input[2] != NULL) {
	    fprintf(stderr, "Expected two columns in input\n");
	    return 1;
	}
	freq = input[0];
	mag = input[1];
	if (taps <= 0) {
	    taps = len;
	}
	if (dbinput) {
	    tb_dbconvert(mag, len, false);
	}
	if (fdes == FILTER_LINEARPHASE) {
	    out = linear_phase_filter(freq, mag, len, taps, sfreq);
	} else {
	    out = minimum_phase_filter(freq, mag, len, taps, sfreq);
	}
	break;
	
	
    case FILTER_ALLPASS:
	if (n < argc - 1) {
	    if ((stream = fopen(argv[++n], "rt")) == NULL) {
		fprintf(stderr, "Could not open file \"%s\" for reading: %s\n",
			argv[n], strerror(errno));
		return 1;
	    }
	}
	if ((input = coeffs_parse(&len, stream)) == NULL) {
	    fprintf(stderr, "Failed to parse phase response\n");
	    return 1;
	}
	fclose(stream);
	if (input[1] == NULL || input[2] != NULL) {
	    fprintf(stderr, "Expected two columns in input\n");
	    return 1;
	}
	freq = input[0];
	phase = input[1];
	if (taps <= 0) {
	    taps = len;
	}
	if (degreeinput) {
	    tb_degreeconvert(phase, len, false);
	}
	out = allpass_filter(freq, phase, len, taps, sfreq);
	break;

	
    case FILTER_CONVOLVE:
	if (n == argc - 3) {
	    if ((stream = fopen(argv[++n], "rt")) == NULL) {
		fprintf(stderr, "Could not open file \"%s\" for reading: %s\n",
			argv[n], strerror(errno));
		return 1;
	    }	    
	    if ((input = coeffs_parse(&len, stream)) == NULL) {
		fprintf(stderr, "Failed to parse first impulse response\n");
		return 1;
	    }
	    fclose(stream);
	    if (input[1] != NULL) {
		fprintf(stderr, "Expected only one column in first impulse "
			"response\n");
		return 1;
	    }
	    if ((stream = fopen(argv[++n], "rt")) == NULL) {
		fprintf(stderr, "Could not open file \"%s\" for reading: %s\n",
			argv[n], strerror(errno));
		return 1;
	    }	    
	    if ((input2 = coeffs_parse(&len2, stream)) == NULL) {
		fprintf(stderr, "Failed to parse second impulse response\n");
		return 1;
	    }
	    fclose(stream);
	    if (input2[1] != NULL) {
		fprintf(stderr, "Expected only one column in second impulse "
			"response\n");
		return 1;
	    }
	    if (len2 > len) {
		n = len;
		len = len2;
		len2 = n;
		imp = input2[0];
		imp2 = input[0];
	    } else {
		imp = input[0];
		imp2 = input2[0];		
	    }
	} else {
	    if (n == argc - 2) {
		if ((stream = fopen(argv[++n], "rt")) == NULL) {
		    fprintf(stderr, "Could not open file \"%s\" for "
			    "reading: %s\n", argv[n], strerror(errno));
		    return 1;
		}
	    }
	    if ((input = coeffs_parse(&len, stream)) == NULL) {
		fprintf(stderr, "Failed to parse impulse responses\n");
		return 1;
	    }
	    fclose(stream);
	    if (input[1] == NULL || input[2] != NULL) {
		fprintf(stderr, "Expected two columns in input\n");
		return 1;
	    }
	    imp = input[0];
	    imp2 = input[1];
	    len2 = len;
	}
	if (taps <= 0) {
	    taps = len + len2 - 1;
	}
	if (taps > len) {
	    if ((out = malloc(taps * sizeof(double))) == NULL) {
		fprintf(stderr, "Memory allocation error\n");
		return 1;
	    }
	    memset(out, 0, taps * sizeof(double));
	    memcpy(out, imp, len * sizeof(double));
	    imp = out;
	}
	if ((out = malloc(taps * sizeof(double))) == NULL) {
	    fprintf(stderr, "Memory allocation error\n");
	    return 1;
	}
	convolve(out, imp, taps, imp2, len2);
	break;

	
    default:
	fprintf(stderr, "Reached default case in switch - this should never "
		"happen\n");
	return 1;
    }
    
    if (n != argc - 1) {
	fprintf(stderr, "Too many parameters\n");
	return 1;
    }
    if (out == NULL) {
	fprintf(stderr, "Could not calculate filter\n");
	return 1;
    }

    if (gain != 0.0) {
	gain = pow(10.0, gain / 20.0);
	for (n = 0; n < taps; n++) {
	    out[n] *= gain;
	}
    }
    if (apply_hamming) {
	if ((win = hamming_window(taps)) == NULL) {
	    fprintf(stderr, "Could not apply hamming window\n");
	    return 1;
	}
	for (n = 0; n < taps; n++) {
	    out[n] *= win[n];
	}	
    }
    
    /* write output */
    if (rawoutput) {
	if (rawsingle) {
	    outsize = taps * sizeof(float);
	    if ((f = malloc(outsize)) == NULL) {
		fprintf(stderr, "Memory allocation failure\n");
		return 1;
	    }
	    for (n = 0; n < taps; n++) {
		f[n] = (float)out[n];
	    }	    
	} else {
	    outsize = taps * sizeof(double);
	}
	if (fwrite(f, outsize, 1, stdout) != 1) {
	    fprintf(stderr, "output failed: %s\n", strerror(errno));
	    return 1;
	}
    } else {
	for (n = 0; n < taps; n++) {
	    printf("%.16e\n", out[n]);
	}
    }
    fprintf(stderr, "Finished!\n");
    return 0;
}
