/*
 * (c) Copyright 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <math.h>
#include <gsl/gsl_fft_real.h>

#include "spline.h"
#include "coeffs.h"
#include "toolbox.h"

#define MAX_CHANNELS 256
#define BUFSIZE 1024
#define PI 3.141592654

#define PRESENTATION_STRING \
"\n" \
"I M P P R O C -- Impulse Response Processing Program        Experimental version\n"\
"====================================================           (c) Anders Torger\n"\
"                                                                    October 2000\n"\
"\n"

#define USAGE_STRING \
"Usage: %s [-ehmprw] [-f <input filename>] [-q <sampling frequency>]\n"\
"\n"\
" -e get only the excess phase part.\n"\
" -h get power response.\n"\
" -m get only the minimum phase part.\n"\
" -p get power response in decibel.\n"\
" -r output as text in hexadecimal memory dump format.\n"\
" -w get phase response.\n"\
" -f <input filename> file to read data from. Default: read from stdin\n"\
" -q <sampling frequency> sampling frequency in Hz of input. Default: 44100\n"\
"\n"\

bool_t
get_phase_response(double phase[],         /* output */
		   double imp[],           /* the impulse response */
		   int len)                /* length (must be a power of two) */
{
    int n;
    double *tmp = malloc(len * sizeof(double));
    
    if (tmp == NULL) {
	fprintf(stderr, "get_phaseresponse: memory allocation failed\n");
	return false;
    }
    
    memcpy(tmp, imp, len * sizeof(double));
    gsl_fft_real_radix2_transform(tmp, 1, len);
    phase[0] = atan2(0, tmp[0]);
    for (n = 1; n < len / 2; n++) {
	phase[n] = atan2(tmp[len-n], tmp[n]);
	phase[len-n] = atan2(-tmp[len-n], tmp[n]);
    }
    phase[len/2] = atan2(0, tmp[len/2]);
    free(tmp);    
    return true;
}

bool_t
get_phase_component(double out[],  /* output (size: len, may be same as imp) */
		    double imp[],  /* the impulse response */
		    int len,       /* length (must be a power of two) */
		    bool_t get_mp) /* true if minimum phase, else excess */
{
    int n;
    gsl_complex_packed_array complex = malloc(len * sizeof(gsl_complex));
    double *tmp = malloc(len * sizeof(double));
    double *tmp1 = malloc(len * sizeof(double));

    if (tmp == NULL || tmp1 == NULL || complex == NULL) {
	fprintf(stderr, "get_phase_component: memory allocation error\n");
	return false;
    }
    for (n = 0; n < len; n++) {
	complex[2*n] = imp[n];
	complex[2*n+1] = 0;
    }
    if (get_mp) {
	if (!tb_real_cepstrum(tmp, tmp1, out, complex, len)) {
	    fprintf(stderr, "get_phase_component: cepstrum calculation "
		    "failed\n");
	    return false;
	}
    } else {
	if (!tb_real_cepstrum(tmp, out, tmp1, complex, len)) {
	    fprintf(stderr, "get_phase_component: cepstrum calculation "
		    "failed\n");
	    return false;
	}
    }
    free(complex);
    free(tmp);
    free(tmp1);
    return true;
}

int
main(int argc, char *argv[])
{
    bool_t calc_power_response = false;
    bool_t raw_output = false, calc_phase_response = false;
    bool_t decibel_output = false, done_fft = false;
    bool_t extract_minimum_phase = false, extract_excess_phase = false;
    int len, order = 0, n, i;
    double sfreq = 44100, *imp, **input, dtmp;
    FILE *instream = stdin;
    uint64_t tmp, tmp1;

    fprintf(stderr, PRESENTATION_STRING);

    /* parse arguments */
    if (argc < 2) {
	fprintf(stderr, USAGE_STRING, argv[0]);
	return 1;
    }
    for (n = 1; n < argc && argv[n][0] == '-'; n++) {
	switch (argv[n][1]) {
	case 'e':
	    extract_excess_phase = true;
	    break;
	case 'h':
	    calc_power_response = true;
	    break;
	case 'm':
	    extract_minimum_phase = true;
	    break;
	case 'p':
	    calc_power_response = true;
	    decibel_output = true;
	    break;
	case 'r':
	    raw_output = true;
	    break;
	case 'w':
	    calc_phase_response = true;
	    break;
	case 'f':
	    if ((instream = fopen(argv[++n], "rb")) == NULL) {
		fprintf(stderr, "Could not open \"%s\" for reading\n.",
			argv[n]);
		return 1;
	    }
	    break;
	case 'q':
	    sfreq = atof(argv[++n]);
	    if (sfreq <= 0) {
		fprintf(stderr, "Invalid sample frequency \"%s\"\n", argv[n]);
		return 1;
	    }
	    break;
	default:
	    fprintf(stderr, "Invalid argument \"%s\"\n\n", argv[n]);
	    fprintf(stderr, USAGE_STRING, argv[0]);
	    return 1;
	}
    }
    if (extract_minimum_phase && extract_excess_phase) {
	fprintf(stderr, "Cannot extract minimum phase and excess phase at the "
		"same time.\n");
	return 1;
    }
    if (calc_power_response && calc_phase_response) {
	fprintf(stderr, "Cannot calculate power and phase response at the "
		"same time.\n");
	return 1;
    }
    
    /* get the input data into memory */
    fprintf(stderr, "reading data from input...");
    if ((input = coeffs_parse(&len, instream)) == NULL) {
	fprintf(stderr, "Failed to parse impulse response\n");
	return 1;
    }
    if (input[1] != NULL) {
	fprintf(stderr, "More than one column in input\n");
	return 1;
    }
    if (instream != stdin) {
	fclose(instream);
    }
    if (tb_getlog2(len) == -1) {
	for (n = 0, i = len; n < 32; i >>= 1, n++) {
	    if ((i & 1) != 0) {
		order = n;
	    }
	}
	order++;
    } else {
	order = tb_getlog2(len);
    }
    
    if ((imp = malloc((1 << order) * sizeof(double))) == NULL) {
	fprintf(stderr, "Failed to allocate memory\n");
	return 1;
    }
    bzero(imp + len, ((1 << order) - len) * sizeof(double));
    memcpy(imp, input[0], len * sizeof(double));
    coeffs_free(input);
    fprintf(stderr, "finished\n");

    if (extract_minimum_phase) {
	fprintf(stderr, "extracting minimum phase component...");
	if (!get_phase_component(imp, imp, 1 << order, true)) {
	    return 1;
	}
	fprintf(stderr, "finished\n");	
    }
    if (extract_excess_phase) {
	fprintf(stderr, "extracting excess phase component...");
	if (!get_phase_component(imp, imp, 1 << order, false)) {
	    return 1;
	}
	fprintf(stderr, "finished\n");	
    }

    if (calc_power_response) {
	fprintf(stderr, "calculating power response...");
	gsl_fft_real_radix2_transform(imp, 1, 1 << order);
	tb_abs_on_halfcomplex(imp, 1 << order);
	if (decibel_output) {
	    tb_dbconvert(imp, 1 << order, true);
	}
	fprintf(stderr, "finished\n");
	done_fft = true;
    }

    if (calc_phase_response) {
	fprintf(stderr, "calculating phase response...");
	if (!get_phase_response(imp, imp, 1 << order)) {
	    return 1;
	}
	fprintf(stderr, "finished\n");	
	done_fft = true;
    }

    /* print output */
    fprintf(stderr, "writing output...");
    if (raw_output && !done_fft) {
	/* impulse response in raw text format */
	for (n = 0; n < (1 << order) - 1; n++) {
	    memcpy(&tmp, &imp[n], sizeof(uint64_t));
	    printf("0x%.16llX\n", tmp);
	}
    } else if (!done_fft) {
	/* impulse response in text format */	
	for (n = 0; n < (1 << order) - 1; n++) {
	    printf("%.16e\n", imp[n]);
	}
    } else if (raw_output) {
	/* power or phase response in raw text format */
	for (n = 1; n < 1 << (order - 1); n++) {
	    dtmp = (double)n * sfreq / (double)(1 << order);
	    memcpy(&tmp, &dtmp, sizeof(uint64_t));
	    memcpy(&tmp1, &imp[n], sizeof(uint64_t));
	    printf("0x%.16llX 0x%.16llX\n", tmp, tmp1);
	}
    } else {
	/* power or phase response in text format */
	for (n = 1; n < 1 << (order - 1); n++) {
	    printf("%.16e %.16e\n", (double)n * sfreq / (double)(1 << order),
		   imp[n]);
	}
    }
    fprintf(stderr, "finished!\n");
    return 0;
}
