/*
 * (c) Copyright 1999 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>

#include "mls.h"
    
static int mls_taps[2][24][4] = {
    {{-1,-1,-1,-1},
     { 0, 1,-1,-1},
     { 1, 2,-1,-1},
     { 2, 3,-1,-1},
     { 2, 4,-1,-1},
     { 4, 5,-1,-1},
     { 5, 6,-1,-1},
     { 1, 2, 4, 7},
     { 4, 8,-1,-1},
     { 6, 9,-1,-1},
     { 8,10,-1,-1},
     { 5, 7,10,11},
     { 8, 9,11,12},
     { 3, 7,12,13},
     {13,14,-1,-1},
     { 3,12,14,15},
     {13,16,-1,-1},
     {10,17,-1,-1},
     {13,16,17,18},
     {16,19,-1,-1},
     {18,20,-1,-1},
     {20,21,-1,-1},
     {17,22,-1,-1},
     {16,21,22,23}},
    
    {{-1,-1,-1,-1},
     { 1, 0,-1,-1},
     { 2, 0,-1,-1},
     { 3, 0,-1,-1},
     { 4, 1,-1,-1},
     { 5, 0,-1,-1},
     { 6, 0,-1,-1},
     { 7, 6, 4, 0},
     { 8, 3,-1,-1},
     { 9, 6,-1,-1},
     {10, 1,-1,-1},
     {11, 6, 3, 2},
     {12, 3, 2, 0},
     {13,11,10, 0},
     {14, 7,-1,-1},
     {15, 4, 2, 1},
     {13,16,-1,-1},
     {17, 6,-1,-1},
     {18, 5, 4, 0},
     {19, 2,-1,-1},
     {20, 1,-1,-1},
     {21, 0,-1,-1},
     {22, 4,-1,-1},
     {23, 3, 2, 0}}
};

uint32_t *
maximum_length_sequence(unsigned int order, unsigned int tapset)
{
    int n, bitsum, ntaps, i, ti = order - 1;
    uint32_t *mls, p = 1;

    if (order > 24 || order == 0) {
	fprintf(stderr, "maximum_length_sequence: invalid order (%u). "
		"valid range is 1 - 24\n", order);
	return NULL;
    }
    if (tapset > 1) {
	fprintf(stderr, "maximum_length_sequence: invalid tap set (%u). "
		"valid range is 0 - 1\n", tapset);
	return NULL;
    }
    n = ((1 << order) / 8 < sizeof(uint32_t)) ? sizeof(uint32_t) :
	(1 << order) / 8;
    if ((mls = malloc(n)) == NULL) {
	fprintf(stderr, "maximum_length_sequence: could not allocate memory\n");
	return NULL;
    }
    bzero(mls, n);

    for (ntaps = 0; ntaps < 4 && mls_taps[tapset][ti][ntaps] != -1; ntaps++);
    for (n = 0; n < (1 << order) - 1; n++) {
	for (i = 0, bitsum = 0; i < ntaps; i++) {
	    bitsum += (p >> (mls_taps[tapset][ti][i])) & 1;
	}
	p = (p << 1) + bitsum % 2;
	mls[n/32] = mls[n/32] | (p & 1) << (n % 32);
    }
    return mls;
}

bool_t
fast_hadamard_transform(int64_t signal[],
			int len)
{
    int pow2, n, i, k, bsize, dist;
    int64_t tmp;
    
    if ((len & (len - 1)) != 0) {
	fprintf(stderr, "fast_hadamard_transform: length must be a power "
		"of 2\n");
	return false;
    }
    if (signal == NULL) {
	fprintf(stderr, "fast_hadamard_transform: input signal is NULL\n");
	return false;
    }

    for (n = 0, bsize = 1, pow2 = len; pow2 != 1; n++, pow2 = pow2 >> 1) {
	dist = bsize;
	bsize += bsize;
	for (i = 0; i < len; i += bsize) {
	    for (k = 0; k < dist; k++) {
		tmp = signal[i+k];
		signal[i+k] = tmp + signal[i+k+dist];
		signal[i+k+dist] = tmp - signal[i+k+dist];
	    }
	}
    }
    return true;
}

bool_t
fht(int64_t signal[],
    int len)
{
    int ihalf = len / 2, pow2;
    int irow, i2;
    int64_t *row = malloc(len * sizeof(int64_t));
    
    for (pow2 = len; pow2 != 1; pow2 = pow2 >> 1) {
	for (irow = 0; irow < len; irow++) {
	    if (irow < ihalf) {
		i2 = 2 * irow;
		row[irow] = signal[i2] - signal[i2+1];
	    } else {
		i2 = 2 * (irow - ihalf);
		row[irow] = signal[i2] - signal[i2+1];
	    }
	}
	for (irow = 0; irow < len; irow++) {
	    signal[irow] = row[irow];
	}
    }
    return true;
}
