/*
 * (c) Copyright 1999, 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <math.h>

#include "sample.h"
#include "mls.h"

#define MAX_CHANNELS 256
#define BUFSIZE 1024
#define PI 3.141592654

#define PRESENTATION_STRING \
"\n" \
"M L S 2 I M P -- MLS Response Processing Program           Experimental version\n"\
"================================================              (c) Anders Torger\n"\
"   Converts MLS response to impulse response                        August 2000\n"\
"\n"

#define USAGE_STRING \
"Usage: %s [-r] [-b <input format>] [-c <n channels>,<target channel>]\n"\
"       [-f <input filename>] [-l <left rotate>] [-o <n seconds>]\n"\
"       [-p <n samples>] [-s <tap set index>] <MLS order>\n"\
"\n"\
" -r output as text in hexadecimal memory dump format.\n"\
" -b <input format>, <bits><s(igned)/u(nsigned)><bytes><l(ittle)/b(ig) endian>\n"\
"    <left shift>. Default: 16s2l0\n"\
" -c <n channels>,<target channel> number of interleaved channels and which of\n"\
"    them that holds the MLS response to be processed. Default: 1,1\n"\
" -f <input filename> file to read data from. Default: read from stdin\n"\
" -l <left rotate> rotate the impulse response to the left. Default: 0\n"\
" -o <n samples> number of samples (per channel) to skip in the beginning.\n"\
"    Default: 0\n"\
" -p <n samples> width of pulse in samples of original MLS. Default: 1\n"\
" -s <tap set index> must be same tap set as the original MLS. Default: 0\n"\
"\n"\

#define GETSIGNAL_GETSAMPLE()                                                  \
    longsample = 0;                                                            \
    for (k = 0; k < pulse_width; k++) {                                        \
        longsample += sample_raw2int(sf, &buf[n_channels * sf->bytes * (i+k) * \
				     pulse_width + (channel-1) * sf->bytes]);  \
    }                                                                          \
    data[n*BUFSIZE+i] = longsample;


/* read signal from audio data into 64 bit array */
bool_t
getsignal(int64_t data[],  /* input signal destination */
	  int n_channels,  /* how many channels the signal data is */
	  int channel,     /* channel to read data from, 1 being the first */
	  struct sample_format * sf,
	  int len,         /* how many samples to read per period */
	  int pulse_width, /* width of MLS pulse */
	  FILE *instream)  /* the stream where signal data is found */
{
    uint8_t *buf;
    int64_t longsample;
    int n, i, k, bufsize;

    bufsize = n_channels * sf->bytes * pulse_width * BUFSIZE;
    if ((buf = malloc(bufsize)) == NULL) {
	fprintf(stderr, "getsignal: could not allocate memory for input "
		"buffer (%d bytes)\n", bufsize);
	return false;
    }

    for (n = 0; n < len / BUFSIZE; n++) {
	if (fread(buf, bufsize, 1, instream) != 1) {
	    fprintf(stderr, "failed to read data from input stream\n");
	    free(buf);
	    return false;
	}
	for (i = 0; i < BUFSIZE; i++) {
	    GETSIGNAL_GETSAMPLE();
	}
    }
    if (len % BUFSIZE != 0) {
	if (fread(buf, n_channels * sf->bytes * (len % BUFSIZE), 1, instream)
	    != 1)
	{
	    fprintf(stderr, "failed to read data from input stream\n");
	    free(buf);
	    return false;
	}
	for (i = 0; i < len % BUFSIZE; i++) {
	    GETSIGNAL_GETSAMPLE();
	}
    }
    free(buf);
    return true;
}

#undef GETSIGNAL_GETSAMPLE

/*
 * Convert MLS response to impulse response with help of the Fast Hadamard
 * Transform.
 * The method is clearly described in the paper "Impulse-Response and
 * Reverbation-Decay Measurements Made by Using a Periodic Pseudorandom
 * Sequence" by W. T. Chu (Applied Acoustics v 29 n 3 1990 p 193 - 205)
 */
bool_t
mlsresp2impresp(int64_t *maxval,   /* will receive maximum value of output */
		int64_t data[],    /* MLS response, receives the impulse
				    * response. */
		int order,         /* order of the original MLS */
		int tapset,        /* tap set index of the original MLS */
		int left_rotate,   /* further left rotate after auto rotate */
		bool_t lossless)   /* if true division with (1 << order) is not
				    * performed on the output samples */
{
    uint32_t *mls, mask;
    int *perm0 = NULL, *perm1 = NULL, len = (1 << order) - 1, n, i, k;
    int maxval_index, bitpos, nbits;
    int64_t *fht_data, tmp;

    /* recreate original MLS to be able to create permutation arrays */
    if ((mls = maximum_length_sequence(order, tapset)) == NULL) {
	fprintf(stderr, "mlsresp2impresp: Could not generate MLS\n");
	return false;
    }

    /* create permutation arrays */
    if ((perm0 = malloc(len * sizeof(int))) == NULL ||
	(perm1 = malloc(len * sizeof(int))) == NULL)
    {
	fprintf(stderr, "mlsresp2impresp: could not allocate memory for "
		"permutation arrays\n");
	free(mls); free(perm0); free(perm1);
	return false;
    }
    bzero(perm0, len * sizeof(int));
    bzero(perm1, len * sizeof(int));

    for (n = 0; n < order; n++) {
	for (i = 0; i < len; i++) {
	    k = (len + i - n) % len;
	    if (((mls[k/32] >> (k % 32)) & 1) == 1) {
		perm0[i] = perm0[i] | (1 << (order - n - 1));
	    }
	}
    }
    for (n = 0, mask = 0; n < len; n++) {
	bitpos = 0;
	for (i = 0, nbits = 0; i < order; i++) {
	    k = (len + n - i) % len;
	    if (((mls[k/32] >> (k % 32)) & 1) == 1) {
		nbits++;
		bitpos = i;
	    }
	}
	if (nbits == 1 && (mask & (1 << bitpos)) == 0) {
	    for (i = 0; i < len; i++) {
		k = (len + n - i) % len;
		if (((mls[k/32] >> (k % 32)) & 1) == 1) {
		    perm1[i] = perm1[i] | (1 << (order - bitpos - 1));
		}
	    }
	    mask = mask | (1 << bitpos);
	    if (mask == (1 << order) - 1) {
		break;
	    }
	}
    }
    free(mls);

    /* reorder data prior to FHT */
    if ((fht_data = malloc((len + 1) * sizeof(int64_t))) == NULL) {
	fprintf(stderr, "mlsresp2impresp:could not to allocate memory\n");
	free(perm0); free(perm1);
	return false;
    }
    
    for (n = 0; n < len; n++) {
	fht_data[perm0[n]] = data[n];
    }
    fht_data[0] = 0;

    /* perform Fast Hadamard Transform */
    fast_hadamard_transform(fht_data, len+1);
    
    /* calculate samplesum of original data */
    for (n = 0, tmp = 0; n < len; n++) {
	tmp += data[n];
    }

    /* reorder data after the FHT and rescale sample values */
    for (n = 0, *maxval = 0, maxval_index = 0; n < len; n++) {
	if (!lossless) {
	    data[n] = (fht_data[perm1[n]] - tmp) / (len+1);
	} else {
	    data[n] = fht_data[perm1[n]] - tmp;
	}
	if (data[n] > 0) {
	    if (data[n] > *maxval) {
		*maxval = data[n];
		maxval_index = n;
	    }
	} else if (-data[n] > *maxval) {
	    *maxval = -data[n];
	    maxval_index = n;
	}
    }

    /* find rotation alignment */
    for (n = 1; n < len; n++) {
	if (fabs((double)data[(len + maxval_index - n) % len]) /
	    fabs((double)(*maxval)) < 0.1)
	{
	    for (i = n + len / 10;
		 n < i && fabs((double)data[(len + maxval_index - n) % len]) /
		     fabs((double)(*maxval)) < 0.1;
		 n++);
	    if (n == i) {
		n -= len / 10 + 1;
		n = (len + maxval_index - n) % len;
		break;
	    }
	}
    }
    n += left_rotate;
    if (n >= len) {
	n = maxval_index;
    }
    
    /* rotate impulse response */
    if (n != 0) {
	memcpy(fht_data, &data[n],
	       (len - n) * sizeof(int64_t));
	memcpy(&fht_data[len - n], data,
	       n * sizeof(int64_t));
	memcpy(data, fht_data, len * sizeof(int64_t));
    }
    
    free(fht_data);
    
    return true;    
}

int
main(int argc, char *argv[])
{
    struct sample_format sf;
    bool_t raw_output = false;
    int n_channels = 1, channel = 1, order, n;
    int pulse_width = 1, offset = 0, tapset = 0, left_rotate = 0;
    double *ddata = NULL;
    FILE *instream = stdin;
    int64_t *data, maxval;
    uint64_t tmp;
    char *p;

    fprintf(stderr, PRESENTATION_STRING);

    /* parse arguments */
    sample_parse_format(&sf, "16s2l0");
    if (argc < 2) {
	fprintf(stderr, USAGE_STRING, argv[0]);
	return 1;
    }
    for (n = 1; n < argc - 1 && argv[n][0] == '-'; n++) {
	switch (argv[n][1]) {
	case 'r':
	    raw_output = true;
	    break;
	case 'b':
	    if (!sample_parse_format(&sf, argv[++n])) {
		return 1;
	    }
	    break;
	case 'c':
	    n_channels = atoi(argv[++n]);
	    if (n_channels < 1) {
		fprintf(stderr, "Invalid number of channels in string "
			"\"%s\".\n", argv[n]);
		return 1;
	    }
	    if ((p = strchr(argv[n], ',')) == NULL) {
		fprintf(stderr, "Missing channel specifier in string "
			"\"%s\".\n", argv[n]);
		return 1;
	    }
	    channel = atoi(++p);
	    if (channel < 1 || channel > n_channels) {
		fprintf(stderr, "Invalid channel specifier in string "
			"\"%s\".\n", argv[n]);
		return 1;
	    }
	    break;
	case 'f':
	    if ((instream = fopen(argv[++n], "rb")) == NULL) {
		fprintf(stderr, "Could not open \"%s\" for reading\n.",
			argv[n]);
		return 1;
	    }
	    break;
	case 'l':
	    if ((left_rotate = atoi(argv[++n])) < 0) {
		fprintf(stderr, "Invalid left rotate \"%s\", it must not be "
			"negative.\n", argv[n]);
		return 1;		
	    }
	    break;
	case 'o':
	    if ((offset = atoi(argv[++n])) < 1) {
		fprintf(stderr, "Invalid offset \"%s\", it must not be "
			"negative.\n", argv[n]);
		return 1;
	    }
	    break;
	case 'p':
	    if ((pulse_width = atoi(argv[++n])) < 1 || pulse_width > 10) {
		fprintf(stderr, "Invalid pulse width \"%s\", valid range is "
			"1 - 10.\n", argv[n]);
		return 1;
	    }
	    break;
	case 's':
	    tapset = atoi(argv[++n]);
	    if (tapset < 0) {
		fprintf(stderr, "Invalid tap set \"%s\"\n", argv[n]);
		return 1;
	    }
	    break;
	default:
	    fprintf(stderr, "Invalid argument \"%s\"\n\n", argv[n]);
	    fprintf(stderr, USAGE_STRING, argv[0]);
	    return 1;
	}
    }
    if (n != argc - 1) {
	fprintf(stderr, "Missing <MLS order> argument\n");
	return 1;
    }
    if ((order = atoi(argv[n])) < 1) {
	fprintf(stderr, "Invalid order \"%s\"\n\n", argv[n]);
	return 1;
    }
    
    /* allocate memory to store the signal prior to fast hadamard transform */
    if ((data = malloc((1 << order) * sizeof(int64_t))) == NULL) {
	fprintf(stderr, "could not allocate memory (%u bytes needed)\n",
		(1 << order) * sizeof(int64_t));
    }
    
    /* get the input data into memory */
    fprintf(stderr, "reading data from input...");
    if (offset != 0) {
	fseek(instream, offset * n_channels * sf.bytes, 0);
    }
    if (!getsignal(data, n_channels, channel, &sf, (1 << order) - 1,
		   pulse_width, instream))
    {
	fprintf(stderr, "Could not read input signal\n");
	return 1;
    }
    if (instream != stdin) {
	fclose(instream);
    }
    fprintf(stderr, "finished\n");

    /* get impulse response */
    fprintf(stderr, "calculating impulse response...");
    if (!mlsresp2impresp(&maxval, data, order, tapset, left_rotate, true))
    {
	fprintf(stderr, "Could not calculate impulse response\n");
	return 1;
    }
    fprintf(stderr, "finished\n");

    /* convert data to double format */
    ddata = (double *)data;
    for (n = 0; n < (1 << order) - 1; n++) {	
	ddata[n] = (double)data[n] / maxval;
    }

    /* print output */
    fprintf(stderr, "writing output...");
    if (raw_output) {
	/* impulse response in raw text format */
	for (n = 0; n < (1 << order) - 1; n++) {
	    memcpy(&tmp, &ddata[n], sizeof(uint64_t));
	    printf("0x%.16llX\n", tmp);
	}
    } else {
	/* impulse response in text format */	
	for (n = 0; n < (1 << order) - 1; n++) {
	    printf("%f\n", ddata[n]);
	}
    }
    fprintf(stderr, "finished!\n");
    return 0;
}
