/*
 * (c) Copyright 1999, 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>

#include "sample.h"
#include "mls.h"

#define PRESENTATION_STRING \
"\n" \
"M L S G E N -- a Maximum Length Sequence Generator         Experimental version\n"\
"==================================================            (c) Anders Torger\n"\
"                                                                      Mars 2000\n"\
"\n"

#define USAGE_STRING \
"Usage: %s [-it] [-a <amplitude>] [-b <output format>]\n"\
"        [-c <n channels>[,<output ch>,...]] [-p <n samples>]\n"\
"        [-r <n times>] [-s <tap set index>] <order>\n"\
"\n"\
" -i use IRS (Inverse Repeated Sequence)\n"\
" -t print sequence as text (once), values will be -1 and 1. Other optional \n"\
"    parameters will be ignored (except -i)\n"\
" -a <amplitude> amplitude for the MLS values, from 0.0 - 1.0. Default: 0.5\n"\
" -b <output format>, <bits><s(igned)/u(nsigned)><bytes><l(ittle)/b(ig) endian>\n"\
"    <left shift>. Default: 16s2l0\n"\
" -c <n channels>[,<output ch>,...] number of channels and in which channel(s)\n"\
"    to play. Default: 1,1\n"\
" -p <n samples> width of pulse in samples. Default: 1\n"\
" -r <n times> number of times to play the output sequence. A value less than 1\n"\
"    means infinite number of times. Default: 1\n"\
" -s <tap set index> choose tap set, to vary an MLS of the same order.\n"\
"    Default: 0\n"\
"\n"\

int
main(int argc, char *argv[])
{
    struct sample_format sf;
    uint8_t *buf;
    uint32_t *mls, zerosample = 0;
    char *p;
    double amplitude = 0.5;
    int n_channels = 1, n_times = 1, pulse_width = 1;    
    int n, i, order, mult, sample, tapset = 0;
    bool_t use_irs = false, textmode = false;
    bool_t *active_channel = NULL, ac[1];

    fprintf(stderr, PRESENTATION_STRING);

    /* parse arguments */
    sample_parse_format(&sf, "16s2l0");
    if (argc < 2) {
	fprintf(stderr, USAGE_STRING, argv[0]);
	return 1;
    }
    for (n = 1; n < argc - 1 && argv[n][0] == '-'; n++) {
	switch (argv[n][1]) {
	case 'i':
	    use_irs = true;
	    break;
	case 't':
	    textmode = true;
	    break;
	case 'a':
	    amplitude = atof(argv[++n]);
	    if (amplitude < 0.0 || amplitude > 1.0) {
		fprintf(stderr, "Invalid amplitude \"%s\". Must be in range "
			"0.0 - 1.0\n", argv[n]);
		return 1;
	    }
	    break;
	case 'b':
	    if (!sample_parse_format(&sf, argv[++n])) {
		return 1;
	    }
	    break;
	case 'c':
	    n_channels = atoi(argv[++n]);
	    if (n_channels < 1) {
		fprintf(stderr, "Invalid number of channels \"%s\"\n",
			argv[n]);
		return 1;
	    }
	    if ((active_channel = malloc(n_channels * sizeof(bool_t)))
		== NULL)
	    {
		fprintf(stderr, "Could not allocate memory for channel info\n");
		return 1;
	    }
	    bzero(active_channel, n_channels * sizeof(bool_t));
	    if ((p = strchr(argv[n], ',')) == NULL) {
		for (i = 0; i < n_channels; i++) {
		    active_channel[i] = true;
		}
	    } else {
		for (; p != NULL; p = strchr(p, ',')) {
		    if ((i = atoi(++p)) < 1 || i > n_channels) {
			fprintf(stderr, "Invalid channel index in -c "
				"parameter\n");
			return 1;
		    }
		    active_channel[i-1] = true;
		}
	    }
	    break;
	case 'p':
	    if ((pulse_width = atoi(argv[++n])) < 1 || pulse_width > 10) {
		fprintf(stderr, "Invalid pulse width \"%s\", valid range is "
			"1 - 10.\n", argv[n]);
		return 1;
	    }
	    break;
	case 'r':
	    n_times = atoi(argv[++n]);
	    break;
	case 's':
	    if ((tapset = atoi(argv[++n])) < 0) {
		fprintf(stderr, "Invalid tap set \"%s\", must be larger than "
			"zero\n", argv[n]);
		return 1;
	    }
	    break;
	default:
	    fprintf(stderr, "Invalid argument \"%s\"\n\n", argv[n]);
	    fprintf(stderr, USAGE_STRING, argv[0]);
	    return 1;
	}
    }
    if (n != argc - 1) {
	fprintf(stderr, "Missing <order> argument\n");
	return 1;
    }
    order = atoi(argv[n]);

    /* generate the maximum length sequence */
    if ((mls = maximum_length_sequence(order, tapset)) == NULL) {
	fprintf(stderr, "could not generate maximum length sequence\n");
	return 1;
    }

    /* if text mode, print sequence once then exit */
    if (textmode) {
	for (n = 0; n < (1 << order) - 1; n++) {
	    printf("%d\n", 1 - (((mls[n/32] >> (n % 32)) << 1) & 2));
	}
	if (use_irs) {
	    for (n = 0; n < (1 << order) - 1; n++) {
		printf("%d\n", (((mls[n/32] >> (n % 32)) << 1) & 2) - 1);
	    }
	}
	fprintf(stderr, "finished!\n");
	return 0;
    }

    if ((buf = malloc(n_channels * sf.bytes)) == 0) {
	fprintf(stderr, "could not allocate memory for output buffer\n");
	return 1;
    }
    mult = (1 << (sf.bits - 1)) * amplitude;
    if (mult >= 1 << (sf.bits - 1)) {
	mult = (1 << (sf.bits - 1)) - 1;
    }
    if (n_times < 1) {
	n_times = -1;
    }
    if (active_channel == NULL) {
	active_channel = ac;
	ac[0] = true;
    }
    /*
     * Output the signal. This loop is quite processor intensive since nothing
     * is buffered. Most often it would be feasible to buffer the whole output
     * sequence... perhaps in a future implementation.
     */
    do {
	do {
	    for (n = 0; n < (1 << order) - 1; n++) {
		sample = mult * (1 - (((mls[n/32] >> (n % 32)) << 1) & 2));
		for (i = 0; i < n_channels; i++) {
		    if (active_channel[i]) {
			sample_int2raw(&sf, &buf[i * sf.bytes], sample);
		    } else {
			sample_int2raw(&sf, &buf[i * sf.bytes], zerosample);
		    }
		}
		for (i = 0; i < pulse_width; i++) {
		    fwrite(buf, n_channels * sf.bytes, 1, stdout);
		}
	    }
	    if (use_irs) {
		mult = -mult;
	    }
	} while (mult < 0);
	if (n_times < 0) {
	    continue;
	}
    } while (--n_times);

    fprintf(stderr, "finished!\n");
    return 0;
}
