/*
 * (c) Copyright 1999 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "spline.h"
#include "coeffs.h"

#define PRESENTATION_STRING \
"\n" \
"M S P R E P R O C  -- a Magnitude Spectrum Pre-processor   Experimental version\n"\
"========================================================      (c) Anders Torger\n"\
"      Designed for Room Equalisation applications                 December 1999\n"\
"\n"

#define USAGE_STRING \
"Usage: %s [-iorst] [-b <low>,<high>] [-c <corr filename>]\n"\
"       [-f <input filename>] [-g <max gain]\n"\
"       [-m <interpolation method>,[resolution]]\n"\
"       [-v <offset>] [-w <low>,<high>] <smoothing method> [<option(s)>]\n"\
"\n"\
" -i input values are in decibel\n"\
" -o output in decibel\n"\
" -r output values in raw hexadecimal format.\n"\
" -s output smoothed magnitude spectrum, instead of the target spectrum.\n"\
" -t input is the actual target spectrum, and not the measured room response.\n"\
" -b <low>,<high> output spectrum bounds in Hz. Default: 0,22050\n"\
" -c <corr filename> file to read correction response from. Values must be in dB.\n"\
" -f <input filename> file to read data from. Default: input read from stdin\n"\
" -g <max gain> maximum gain in decibel for correcting dips. Negative value \n"\
"    means infinity. Default: 3\n"\
" -m <interpolation method>,[output length] choose interpolation method, SPLINE, \n"\
"    LINEAR or NONE. Resolution is in Hz. Default: SPLINE,16384\n"\
" -v <offset> offset input spectrum with <offset> dB. Default: 0\n"\
" -w <low>,<high> output spectrum interesting bandwidth in Hz. Default: 40,16000\n"\
"\n"\
" <smoothing method> [<option(s)>] choose smoothing method from one of the\n"\
" methods descriped below:\n"\
"   SPLIT [split frequency] [low octave fraction] [high octave fraction]\n"\
"     dual octave bandwidths with a split frequency. Default options are\n"\
"     \"200 1/16 1/3\", which means below 200 Hz 1/16 octave smoothing is \n"\
"     performed, above 1/3.\n"\
"   WARP [warp factor] [sampling frequency] [n taps] [maximum resolution]\n"\
"     use bandwidths corresponding (approximately) to the resolution of a warped\n"\
"     filter. Default options are \"127/128 44100 200 1/16\".\n"\
"\n"

#define INT_SPLINE 1
#define INT_LINEAR 2
#define INT_NONE   3

#define SMOOTH_SPLIT 1
#define SMOOTH_WARP  2

#define MAXSMOPT 4

/* convert an array to and from dB */
void
dbconvert(double signal[],
	  int len,
	  bool_t to_db)
{
    int n;

    if (to_db) {
	for (n = 0; n < len; n++) {
	    signal[n] = 20 * log10(signal[n]);
	}
    } else {
	for (n = 0; n < len; n++) {
	    signal[n] = pow(10, signal[n] / 20);
	}
    }
}

/* Perform avaraging on a magnitude spectrum */
double **
mssmooth(int *len,       /* receives length of output */
	 double *ms[],   /* magnitude spectrum */
	 int mslen,      /* length of magnitude spectrum */
	 double fb[],    /* frequency bands */
	 int fblen,      /* length of frequency band array */
	 double maxgain) /* maximum gain of smoothed spectrum (in dB) */
{
    double **s = malloc(3 * sizeof(double *));
    int n, i, k;

    if (len == NULL || ms == NULL || mslen < 1 || fb == NULL || fblen < 1
	|| ms[0] == NULL || ms[1] == NULL)
    {
	fprintf(stderr, "mssmooth: at least one input parameter is invalid\n");
	return NULL;
    }
    
    if (s == NULL ||
	(s[0] = malloc((fblen - 1) * sizeof(double))) == NULL ||
	(s[1] = malloc(2 * (fblen - 1) * sizeof(double))) == NULL)
    {
	if (s != NULL) {
	    free(s[0]); free(s[1]); free(s);
	}
	fprintf(stderr, "mssmooth: could not allocate memory\n");
	return NULL;
    }
    s[2] = NULL;

    if (maxgain >= 0) {
	maxgain = pow(10, maxgain / 20.0);
    }

    for (n = 0; n < mslen && ms[0][n] < fb[0]; n++);
    if (n == mslen) {
	fprintf(stderr, "mssmooth: no frequency bands in spectrum\n");
	free(s[0]); free(s[1]); free(s);
	return NULL;
    }
    for (i = 0; n < mslen && i < fblen - 1; n++, i++) {
	for (k = 0, s[1][i] = 0.0; ms[0][n] < fb[i+1]; n++, k++) {
	    s[1][i] += ms[1][n];
	}
	if (k == 0) {
	    fprintf(stderr, "mssmooth: frequency band too narrow: found no "
		    "value in magnitude spectrum within the band\n");
	    free(s[0]); free(s[1]); free(s);
	    return NULL;
	}
	s[1][i] /= (double)k;              /* magnitude */
	s[0][i] = (fb[i] + fb[i+1]) / 2.0; /* frequency */
	if (s[1][i] > maxgain && maxgain >= 0) {
	    s[1][i] = maxgain;
	}
    }
    *len = i;
    return s;
}

/* warped frequency. f = frequency, wf = warp factor, sf = sampling frequency */
#define FWARP(f, wf, sf)                                                       \
    ((f) + ((sf) / M_PI) * atan((wf) * sin(2 * M_PI * (f) / (sf)) /            \
    (1.0 - (wf) * cos(2 * M_PI * (f) / (sf)))))

int
main(int argc, char *argv[])
{
    bool_t dbinput = false, dboutput = false, rawoutput = false;
    bool_t smoothonly = false, targetinput = false;
    double lowbound = 0.0, highbound = 22050.0, maxgain = 3.0;
    double lowib = 40.0, highib = 16000.0, smopt[MAXSMOPT], offset = 0.0;
    FILE *stream = stdin, *cstream = NULL;
    double **input, *fbands = NULL, *output[3], **smoothed, *tmp[2];
    double *corr, **tmp1;
    int intmethod = INT_SPLINE, smethod, intlen = 16384, intstart, intend;
    int n, i, len, fblen = 0;
    char *p;
    
    fprintf(stderr, PRESENTATION_STRING);

    /* parse arguments */
    if (argc < 2) {
	fprintf(stderr, USAGE_STRING, argv[0]);
	return 1;
    }
    for (n = 1; n < argc - 1 && argv[n][0] == '-'; n++) {
	switch (argv[n][1]) {
	case 'i':
	    dbinput = true;
	    break;
	case 'o':
	    dboutput = true;
	    break;
	case 'r':
	    rawoutput = true;
	    break;
	case 's':
	    smoothonly = true;
	    break;
	case 't':
	    targetinput = true;
	    break;
	case 'b':
	    if ((p = strchr(argv[++n], ',')) == NULL) {
		fprintf(stderr, "Missing comma in bounds option: \"%s\"",
			argv[n]);
		return 1;
	    }
	    *p = '\0';
	    lowbound = atof(argv[n]);
	    highbound = atof(++p);
	    if (lowbound >= highbound) {
		fprintf(stderr, "Invalid bounds in bounds option");
		return 1;
	    }
	    break;
	case 'c':
	    if ((cstream = fopen(argv[++n], "rt")) == NULL) {
		fprintf(stderr, "Failed to open \"%s\" for reading\n", argv[n]);
		return 1;
	    }
	    break;
	case 'f':
	    if ((stream = fopen(argv[++n], "rt")) == NULL) {
		fprintf(stderr, "Failed to open \"%s\" for reading\n", argv[n]);
		return 1;
	    }
	    break;
	case 'g':
	    maxgain = atof(argv[++n]);
	    break;
	case 'm':
	    if ((p = strchr(argv[++n], ',')) != NULL) {
		*p = '\0';
		if ((intlen = atoi(++p)) < 100) {
		    fprintf(stderr, "Interpolation length is invalid or too "
			    "small: \"%s\"", p);
		    return 1;
		}
	    }
	    if (strcasecmp(argv[n], "NONE") == 0) {
		intmethod = INT_NONE;
	    } else if (strcasecmp(argv[n], "SPLINE") == 0) {
		intmethod = INT_SPLINE;
	    } else if (strcasecmp(argv[n], "LINEAR") == 0) {
		intmethod = INT_LINEAR;
	    } else {
		fprintf(stderr, "Invalid interpolation option: \"%s\"\n",
			argv[n]);
		return 1;		
	    }
	    
	    break;
	case 'v':
	    offset = atof(argv[++n]);
	    break;
	case 'w':
	    if ((p = strchr(argv[++n], ',')) == NULL) {
		fprintf(stderr, "Missing comma in interesting bandwidth "
			"option: \"%s\"", argv[n]);
		return 1;
	    }
	    *p = '\0';
	    lowib = atof(argv[n]);
	    highib = atof(++p);
	    if (lowib >= highib) {
		fprintf(stderr, "Invalid bounds in interesting bandwidth "
			"option");
		return 1;
	    }
	    break;
	}
    }
    if (highbound < highib || lowib < lowbound) {
	fprintf(stderr, "bandwidth bounds / interesting bandwidth mismatch\n");
	return 1;
    }
    
    if (n >= argc) {
	fprintf(stderr, "Missing parameter\n");
	return 1;
    }
    if (strcasecmp(argv[n], "SPLIT") == 0) {
	smethod = SMOOTH_SPLIT;
	smopt[0] = 200.0;
	smopt[1] = 1.0 / 16.0;
	smopt[2] = 1.0 / 3.0;
	for (i = 0, n += 1; i + n < argc && i < MAXSMOPT; i++) {
	    smopt[i] = coeffs_fractionparse(argv[i+n]);
	}
	if (smopt[0] < 0 || smopt[1] <= 0 || smopt[2] <= 0) {
	    fprintf(stderr, "At least one smoothing option is invalid\n");
	    return 1;
	}
    } else if (strcasecmp(argv[n], "WARP") == 0) {
	smethod = SMOOTH_WARP;
	smopt[0] = 127.0 / 128.0;
	smopt[1] = 44100.0;
	smopt[2] = 200.0;
	smopt[3] = 1.0 / 16.0;
	for (i = 0, n += 1; i + n < argc && i < MAXSMOPT; i++) {
	    smopt[i] = coeffs_fractionparse(argv[i+n]);
	}
	if (smopt[1] <= 0) {
	    fprintf(stderr, "Invalid maximum resolution given\n");
	    return 1;
	}
    } else {
	fprintf(stderr, "Invalid smoothing method: \"%s\"\n",
		argv[n]);
	return 1;		
    }
    
    /* read input */
    if ((input = coeffs_parse(&len, stream)) == NULL) {
	fprintf(stderr, "Failed to parse input\n");
	return 1;
    }
    if (input[1] == NULL) {
	fprintf(stderr, "Frequency or magnitude column missing in input\n");
	return 1;
    }
    if (dbinput) {
	dbconvert(input[1], len, false);	
    }
    if (offset != 0) {
	offset = pow(10, offset / 20);
	for (n = 0; n < len; n++) {
	    input[1][n] = input[1][n] * offset;
	}
    }
    if (!targetinput) {
	for (n = 0; n < len; n++) {
	    input[1][n] = 1.0 / input[1][n];
	}
    }
    
    /* sanity-check input */
    if (input[0][0] < 0) {
	fprintf(stderr, "First frequency point in input is negative\n");
	return 1;
    }
    if (input[1][0] < 0) {
	fprintf(stderr, "Negative magnitude value found in input "
		"(a decibel value?)\n");
	return 1;
    }
    for (n = 1; n < len; n++) {
	if (input[0][n-1] >= input[0][n]) {
	    fprintf(stderr, "Input frequency points are not sorted in "
		    "ascending order\n");
	    return 1;
	}
	if (input[1][n] < 0) {
	    fprintf(stderr, "Negative magnitude value found in input "
		    "(a decibel value?)\n");
	    return 1;
	}
    }    
    if (lowib < input[0][0] || highib > input[0][len-1]) {
	fprintf(stderr, "Interesting bandwidth (%f - %f Hz) not covered in "
		"input\n", lowib, highib);
	return 1;
    }

    /* read and apply eventual correction response */
    if (cstream != NULL) {
	if ((tmp1 = coeffs_parse(&n, cstream)) == NULL) {
	    fprintf(stderr, "Failed to parse correction response\n");
	    return 1;
	}
	if ((corr = malloc(len * sizeof(double))) == NULL ||
	    (tmp[0] = malloc((n + 2) * sizeof(double))) == NULL ||
	    (tmp[1] = malloc((n + 2) * sizeof(double))) == NULL)
	{
	    fprintf(stderr, "Could not allocate memory\n");
	    return 1;
	}	
	if (tmp1[0][0] < input[0][0]) {
	    fprintf(stderr, "Correction response lower bound frequency (%f Hz) "
		    "not within input range\n", tmp1[0][0]);
	    return 1;
	}
	if (tmp1[0][n-1] > input[0][len-1]) {
	    fprintf(stderr, "Correction response upper bound frequency (%f Hz) "
		    "not within input range\n", tmp1[0][n-1]);
	    return 1;
	}
	memcpy(&tmp[0][1], tmp1[0], n * sizeof(double));
	memcpy(&tmp[1][1], tmp1[1], n * sizeof(double));
	coeffs_free(tmp1);
	tmp[0][0] = input[0][0];
	tmp[1][0] = 0;
	tmp[0][n+1] = input[0][len-1];
	tmp[1][n+1] = 0;
	if (!linear(corr, input[0], tmp[0], tmp[1], n+2, len)) {
	    fprintf(stderr, "linear interpolation of correction response "
		    "failed\n");
	    return 1;
	}
	free(tmp[0]);
	free(tmp[1]);
	dbconvert(corr, len, false);
	for (n = 0; n < len; n++) {
	    input[1][n] /= corr[n];
	}
	free(corr);
    }    
    
    /* create frequency band array */
    i = 128;
    if ((fbands = malloc(i * sizeof(double))) == NULL) {
	fprintf(stderr, "Could not allocate memory\n");
	return 1;
    }
    switch (smethod) {
    case SMOOTH_SPLIT:
	if (lowib == 0) {
	    fprintf(stderr, "Interesting bandwidth lower bound must be larger "
		"than zero\n");
	    return 1;
	}
	fbands[1] = 0;
	for (n = 1, fbands[0] = lowib; fbands[n-1] < smopt[0]; n++) {
	    fbands[n] = fbands[n-1] + fbands[n-1] * smopt[1];
	    if (n == i - 1) {
		i += 128;
		if ((fbands = realloc(fbands, i * sizeof(double))) == NULL) {
		    fprintf(stderr, "Could not allocate memory\n");
		    return 1;
		}
	    }
	}
	for (; fbands[n-1] < highib; n++) {
	    fbands[n] = fbands[n-1] + fbands[n-1] * smopt[2];
	    if (n == i - 1) {
		i += 128;
		if ((fbands = realloc(fbands, i * sizeof(double))) == NULL) {
		    fprintf(stderr, "Could not allocate memory\n");
		    return 1;
		}
	    }
	}
	fbands[n-1] = highib;
	fblen = n;
	break;
    case SMOOTH_WARP:
	/* calculate widths of the warped filter's bands (approximation) */
	for (n = 1, fbands[0] = lowib; fbands[n-1] < highib; n++) {
	    fbands[n] = fbands[n-1] + 0.89 * smopt[1] / smopt[2] *
		(1.0 - smopt[0] * smopt[0]) /
		(1.0 + smopt[0] * smopt[0] + 2 * smopt[0] *
		 cos(2 * M_PI * FWARP(fbands[n-1], smopt[0], smopt[1]) /
		     smopt[1]));
	    if (n == i - 1) {
		i += 128;
		if ((fbands = realloc(fbands, i * sizeof(double))) == NULL) {
		    fprintf(stderr, "Could not allocate memory\n");
		    return 1;
		}
	    }
	}
	fbands[n-1] = highib;
	fblen = n;
	
	/* join bands that are too narrow */
	for (n = 1, i = 0; n < fblen; n++) {
	    if (fbands[n] - fbands[i] >= smopt[3] * fbands[i]) {
		i++;
		fbands[i] = fbands[n];
	    }
	}	
	fbands[i-1] = highib;
	fblen = i;
	break;
    }

    /* create smoothed frequency array */
    if ((smoothed = mssmooth(&len, input, len, fbands, fblen, maxgain))
	== NULL)
    {
	fprintf(stderr, "Smoothing failed\n");
	return 1;
    }    
    coeffs_free(input);
    free(fbands);
    
    /* interpolate intermidiate frequency points */
    output[2] = NULL;
    if (intmethod != INT_NONE) {
	/* insert extra frequency points at start and end of smoothed array */
	if ((tmp[0] = malloc((len + 2) * sizeof(double))) == NULL ||
	    (tmp[1] = malloc((len + 2) * sizeof(double))) == NULL)
	{
	    fprintf(stderr, "Could not allocate memory for output buffer\n");
	    return 1;
	}
	memcpy(&tmp[0][1], smoothed[0], len * sizeof(double));
	memcpy(&tmp[1][1], smoothed[1], len * sizeof(double));
	coeffs_free(smoothed);
	smoothed = tmp;
	len += 2;
	smoothed[0][0] = lowib;
	smoothed[1][0] = 1.0;
	smoothed[0][len-1] = highib;
	smoothed[1][len-1] = 1.0;

	/* do interpolation */
	if ((output[0] = malloc(intlen * sizeof(double))) == NULL ||
	    (output[1] = malloc(intlen * sizeof(double))) == NULL)
	{
	    fprintf(stderr, "Could not allocate memory for output buffer\n");
	    return 1;
	}
	for (n = 0; n < intlen; n++) {
	    output[0][n] = lowbound + (double)n * (highbound - lowbound) /
		(double)intlen;
	}
	for (intstart = 0; output[0][intstart] < lowib; intstart++) {
	    output[1][intstart] = 1.0;
	}
	output[0][0] = lowbound;
	output[0][intlen-1] = highbound;
	for (intend = intlen - 1; output[0][intend] > highib; intend--) {
	    output[1][intend] = 1.0;
	}
	switch (intmethod) {
	case INT_SPLINE:
	    if (!spline(&output[1][intstart], &output[0][intstart], smoothed[0],
			smoothed[1], len, intend - intstart + 1))
	    {
		fprintf(stderr, "Spline interpolation failed\n");
		return 1;
	    }
	    break;
	case INT_LINEAR:
	    if (!linear(&output[1][intstart], &output[0][intstart], smoothed[0],
			smoothed[1], len, intend - intstart + 1))
	    {
		fprintf(stderr, "Linear interpolation failed\n");
		return 1;
	    }
	    break;
	}
	len = intlen;
    } else {
	output[0] = smoothed[0];
	output[1] = smoothed[1];
    }

    /* print output */
    if (smoothonly) {
	for (n = 0; n < len; n++) {
	    output[1][n] = 1.0 / output[1][n];
	}
    }
    if (dboutput) {
	dbconvert(output[1], len, true);
    }
    coeffs_print(stdout, output, len, rawoutput);
    return 0;
}
