/*
 * (c) Copyright 1999, 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <inttypes.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>

#include "defs.h"
#include "sample.h"
#include "filter.h"
#include "wfir.h"
#include "fir.h"
#include "nwfiir_io.h"
#include "nwfiir.h"

#define PRESENTATION_STRING \
"\n"\
"N - W F I I R -- a real-time filter engine for audio       Experimental version\n"\
"====================================================          (c) Anders Torger\n"\
"   Designed for High-Fidelity audio applications                    August 2000\n"\
"\n"\

#define USAGE_STRING \
"Usage: %s [-dpn] [-b <in>,<out>] [-f <frequency>] [-i <card>,<device>,<fs>]\n"\
"          [-o <card>,<device>,<fs>] [-s <fragment size>,<n fragments>]\n"\
"          [[<filter type[:options]> <coeffs/data file> <gain in dB>] ...]\n"\
"\n"\
" -d start with the filter deactivated.\n"\
" -p show progress.\n"\
" -n do nothing, just initialise and exit.\n"\
" -b <in>,<out> format for input and output. Default: 16s2l0,24s4l8\n"\
"    Sample format string: <bits><s/u><bytes><l/b><left shift>\n"\
"    <bits> bits per sample 16 - 32, <s/u> signed/unsigned <l/b> little/big\n"\
"    endian. <bytes> number of bytes per sample 2 - 4 (may be larger than\n"\
"    bits/8). <left shift> how high up the sample bits are stored in the sample\n"\
"    bytes.\n"\
"    Note: multichannel inputs must store samples interleaved. Output samples\n"\
"          are always stored in interleaved order.\n"\
" -f <frequency> sample frequency in Hz. Default: 44100\n"\
" -i <card>,<device>,<fs> use ALSA card for input, fs = hardware fragment size.\n"\
" -o <card>,<device>,<fs> use ALSA card for output, fs = hardware fragment size.\n"\
" -s <fragment size>,<n fragments> size of a single buffer fragment in frames,\n"\
"    (1 frame = 1 sample * n channels) and how many fragments that make up the \n"\
"    whole buffer. Default: 16384,16\n" \
"    For realtime operation, use many small fragments, for non-realtime use few\n" \
"    large fragments.\n" \
"\n" \
" Filter types: WFIR, FIR, IIR and WIIR (only WFIR and FIR are implemented).\n"\
" WFIR/WIIR options\n"\
"   :<warp factor> is 1/2 (0.5), 3/4 (0.75), 7/8 (0.875), 15/16 (0.9375), \n"\
"   31/32 (0.96875), 63/64 (0.984375), 127/128 (0.9921875) or\n"\
"   255/256 (0.99609375). Default: 63/64\n"\
" FIR options (the FIR filter requires MMX)\n"\
"   :<output filename> where to put the generated FIR library from the given\n"\
"   coefficient file. If this option is not given, it is assumed that the\n"\
"   coefficient file parameter is a previously generated FIR library.\n"\
"\n"\
" send SIGHUP (kill -HUP) to the running process to toggle filter processing\n"\
"\n"

static filter_t *filter;
static bool_t active = true;
static struct io_func io_func;

void
nwfiir_abort_all_processes(void)
{
    filter_delete(filter);
    exit(0);
}

static void
sighandler(int signum)
{
    
    
    switch (signum) {
    case SIGCHLD:
	filter_delete(filter);
	exit(0);
	break;
	
    case SIGHUP:
	filter_toggle_processing(filter);
	active = !active;
	if (active) {
	    fprintf(stderr, "\nFilter processing is now ON\n");
	} else {
	    fprintf(stderr, "\nFilter processing is now OFF\n");
	}	
	break;
    default:
    }
}

int
main(int argc,
     char *argv[])
{
    int taps, n_channels, n, alsa_card_in = -1, alsa_dev_in = -1, n_frags = 16;
    int frequency = 44100, fragsize = 16384, alsa_fs_in = -1, alsa_fs_out = -1;
    int alsa_card_out = -1, alsa_dev_out = -1, groups;
    double gain, error, warp_factor;
    bool_t show_progress = false, do_nothing = false;
    filterproc_t *filterproc[MAX_CHANNELS];
    struct sample_format insf, outsf;
    char *p, *firlibname = NULL;

    fprintf(stderr, PRESENTATION_STRING);

    /* parse options arguments */
    if (argc < 4) {	
	fprintf(stderr, USAGE_STRING, argv[0]);
	return 1;
    }

    sample_parse_format(&insf, "16s2l0");
    sample_parse_format(&outsf, "24s4l8");
    for (n = 1; n < argc - 2 && argv[n][0] == '-'; n++) {
	switch (argv[n][1]) {
	case 'd':
	    active = false;
	    break;
	case 'p':
	    show_progress = true;
	    break;
	case 'n':
	    do_nothing = true;
	    break;
	case 'i':
	    if ((p = strchr(argv[n+1], ',')) == NULL) {
		fprintf(stderr, "Missing comma in alsa input config string\n");
		return 1;
	    }
	    *p = '\0';
	    alsa_card_in = atoi(argv[n+1]);
	    alsa_dev_in = atoi(++p);
	    if ((p = strchr(p, ',')) == NULL) {
		fprintf(stderr, "Missing comma in alsa input config string\n");
		return 1;
	    }
	    alsa_fs_in = atoi(++p);
	    n++;
	    break;
	case 'o':
	    if ((p = strchr(argv[n+1], ',')) == NULL) {
		fprintf(stderr, "Missing comma in alsa output config string\n");
		return 1;
	    }
	    *p = '\0';
	    alsa_card_out = atoi(argv[n+1]);
	    alsa_dev_out = atoi(++p);
	    if ((p = strchr(p, ',')) == NULL) {
		fprintf(stderr, "Missing comma in alsa input config string\n");
		return 1;
	    }
	    alsa_fs_out = atoi(++p);
	    n++;
	    break;
	case 'f':
	    frequency = atoi(argv[n+1]);
	    n++;
	    break;
	case 'b':
	    if ((p = strchr(argv[n+1], ',')) == NULL) {
		fprintf(stderr, "Missing comma in sample format string\n");
		return 1;
	    }
	    *p = '\0';
	    if (!sample_parse_format(&insf, argv[n+1]) ||
		!sample_parse_format(&outsf, ++p))
	    {
		return 1;
	    }
	    n++;
	    break;
	case 's':
	    if ((p = strchr(argv[n+1], ',')) == NULL) {
		fprintf(stderr, "Missing comma in buffer size string\n");
		return 1;
	    }
	    *p = '\0';
	    if ((fragsize = atoi(argv[n+1])) < 128) {
		fprintf(stderr, "Fragment size must at least be 128 samples\n");
		return 1;
	    }
	    if ((n_frags = atoi(++p)) < 4) {
		fprintf(stderr, "There must be at least four fragments\n");
		return 1;
	    }
	    n++;	    
	    break;
	default:
	    fprintf(stderr, USAGE_STRING, argv[0]);
	    return 1;
	}
    }

    /* parse filter channel arguments, and initialise filters */
    for (n_channels = 0; n < argc - 2 && n_channels < MAX_CHANNELS;
	 n += 3, n_channels++)
    {
	gain = atof(argv[n+2]);
	fprintf(stderr, "Channel %d: ", n_channels + 1);
	if (strncasecmp(argv[n], "FIR", 3) == 0) {
	    if (insf.bits > 16) {
		fprintf(stderr, "The FIR filter implementation only supports "
			" 16 bit input samples\n");
		return 1;
	    }
	    if (argv[n][3] == ':') {
		firlibname = &argv[n][4];
		p = argv[n+1];
	    } else {
		firlibname = argv[n+1];
		p = NULL;
	    }
	    if ((filterproc[n_channels] =
		 fir_new(p, firlibname, gain, &taps, &groups, &error)) == NULL)
	    {
		fprintf(stderr, "Failed to create FIR filter\n");
		return 1;
	    }
	    fprintf(stderr, "%d tap FIR filter initialised:\n"
		    "  Filter consists of %d coefficient groups\n"
		    "  Mean squares coefficient requantisation error: %.10f\n",
		    taps, groups, error);
	} else if (strncasecmp(argv[n], "WFIR", 4) == 0) {
	    if (insf.bits > 24) {
		fprintf(stderr, "The WFIR filter implementation only supports "
			"24 bit input samples\n");
		return 1;
	    }
	    if (argv[n][4] == ':') {
		if (strcmp(&argv[n][5], "1/2") == 0) {
		    warp_factor = 1.0 / 2.0;
		} else if (strcmp(&argv[n][5], "3/4") == 0) {
		    warp_factor = 3.0 / 4.0;
		} else if (strcmp(&argv[n][5], "7/8") == 0) {
		    warp_factor = 7.0 / 8.0;
		} else if (strcmp(&argv[n][5], "15/16") == 0) {
		    warp_factor = 15.0 / 16.0;
		} else if (strcmp(&argv[n][5], "31/32") == 0) {
		    warp_factor = 31.0 / 32.0;
		} else if (strcmp(&argv[n][5], "63/64") == 0) {
		    warp_factor = 63.0 / 64.0;
		} else if (strcmp(&argv[n][5], "127/128") == 0) {
		    warp_factor = 127.0 / 128.0;
		} else if (strcmp(&argv[n][5], "255/256") == 0) {
		    warp_factor = 255.0 / 256.0;
		} else {
		    fprintf(stderr, "Invalid WFIR option: \"%s\"\n",
			    &argv[n][5]);
		    return 1;
		}
	    } else {
		warp_factor = 63.0 / 64.0;
	    }
	    if ((filterproc[n_channels] = wfir_new(argv[n+1], warp_factor, gain,
						   &taps, &error))
		 == NULL)
	    {
		fprintf(stderr, "Failed to create WFIR filter\n");
		return 1;
	    }
	    fprintf(stderr, "%d tap WFIR filter initialised:\n"
		    "  Largest coefficient requantisation error %.4f %%\n"
		    "  Warp factor is %.8f\n",
		    taps, error * 100, warp_factor);
	} else if (strcasecmp(argv[n], "IIR") == 0) {
	    fprintf(stderr, "IIR filter is not implemented\n");
	    return 1;
	} else if (strncasecmp(argv[n], "WIIR", 4) == 0) {
	    fprintf(stderr, "WIIR filter is not implemented\n");
	    return 1;
	} else {
	    fprintf(stderr, "Invalid filter type: \"%s\". Supported filters: "
		    "FIR, WFIR, IIR, WIIR\n", argv[n]);
	    return 1;
	}
	
    }
    if (n_channels == 0) {
	fprintf(stderr, "No channels\n");
	return 1;
    }
    
    /* init io */
    n = fragsize;
    if (!init_io(alsa_card_in, alsa_dev_in, alsa_fs_in, alsa_card_out,
		 alsa_dev_out, alsa_fs_out, n_channels, &insf, &outsf,
		 frequency, fragsize, &io_func))
    {
	return 1;
    }
    if (fragsize != n) {
	fprintf(stderr, "Fragsize resized from requested %d samples to %d, to "
		"fit hardware buffers\n", n, fragsize);
    }
    
    /* initialise filter group */
    if ((filter = filter_new(fragsize,
			     n_frags,
			     n_channels,
			     &insf,
			     &outsf,
			     filterproc,
			     &io_func)) == NULL)
    {
	fprintf(stderr, "Filter group initialisation failed\n");
	return 1;	
    }
    if (!active) {
	filter_toggle_processing(filter);
    }
    /* install signal handlers */
    signal(SIGHUP, sighandler);
    signal(SIGCHLD, sighandler);

    if (do_nothing) {
	nwfiir_abort_all_processes();
    }
    /* start filter */
    filter_run(filter, show_progress);    
    return 1;
}
