/*
 * (c) Copyright 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <inttypes.h>
#include <unistd.h>
#include <signal.h>

#include "defs.h"
#include "sample.h"
#include "filter.h"
#include "fdrw.h"
#include "nwfiir_io.h"
#include "alsaio.h"

static int input_size;

static int
stdfillbuf(void *buf)
{
    return fread(buf, 1, input_size, stdin);
}

static void
stdflushbuf(void *buf,
	    int size)
{
    if (fwrite(buf, size, 1, stdout) != 1) {
	fprintf(stderr, "\nOutput failure - %s\n", strerror(errno));
	exit(1);
    }
}

static void
stdstopio(void)
{
    fflush(NULL);
}

#ifdef ALSA
static bool_t
testfragments(int alsa_card,
	      int alsa_dev,
	      bool_t is_input,
	      int framesize,
	      int hw_fragsize,
	      int sw_fragsize)
{
    int minfsz, maxfsz, bufsz;
    
    if (!alsaio_getfragsize(alsa_card, alsa_dev, is_input, &minfsz, &maxfsz,
			    &bufsz))
    {
	return false;
    }
    fprintf(stderr, "ALSA %s device:\n  min fragsize: %d bytes\n  "
	    "max fragsize: %d bytes\n  buffer size: %d bytes\n",
	    (is_input) ? "Input" : "Output", minfsz, maxfsz, bufsz);
    if (sw_fragsize * framesize % hw_fragsize != 0) {
	fprintf(stderr, "Buffer fragment size (%d frames = %d bytes) is "
		"not divisable with %s\nhardware fragment size "
		"(%d bytes)\n",	sw_fragsize, sw_fragsize * framesize,
		(is_input) ? "input" : "output", hw_fragsize);
	return false;
    }
    fprintf(stderr, "  %d hardware fragments per software %s fragment\n",
	    sw_fragsize * framesize / hw_fragsize,
	    (is_input) ? "input" : "output");
    fprintf(stderr, "  %d software %s fragments fits into the hardware "
	    "buffer\n", bufsz / (sw_fragsize * framesize),
	    (is_input) ? "input" : "output");
    if (bufsz / (sw_fragsize * framesize) < 2) {
	fprintf(stderr, "At least two fragments must fit into in the "
		"hardware buffer\n");
	return false;
    }
    return true;
}
#endif

bool_t
init_io(int alsa_card_in,
	int alsa_dev_in,
	int alsa_fs_in,
	int alsa_card_out,
	int alsa_dev_out,
	int alsa_fs_out,
	int n_channels,
	struct sample_format *in_sf,
	struct sample_format *out_sf,
	int rate,
	int fragsize,
	struct io_func *io_func)
{    
    if (alsa_card_out != -1) {
#ifdef ALSA
	if (!testfragments(alsa_card_out, alsa_dev_out, false,
			   out_sf->bytes * n_channels, alsa_fs_out, fragsize))
	{
	    fprintf(stderr, "Requested fragment size is incompatible with "
		    "audio output hardware\n");
	    return false;
	}
	if (!alsaio_init_output(alsa_fs_out, alsa_card_out, alsa_dev_out,
				n_channels, rate, out_sf))
	{
	    fprintf(stderr, "Could not open ALSA device for output\n");
	    return false;
	}
	io_func->flushbuf = alsaio_output;
	io_func->stop_io = alsaio_stop;
#else
	fprintf(stderr, "ALSA support not included in this binary\n");
	return false;
#endif
    } else {
	io_func->flushbuf = stdflushbuf;
	io_func->stop_io = stdstopio;
    }
    
    if (alsa_card_in != -1) {
#ifdef ALSA
	if (!testfragments(alsa_card_in, alsa_dev_in, true,
			   in_sf->bytes * n_channels, alsa_fs_in, fragsize))
	{
	    fprintf(stderr, "Requested fragment size is incompatible with "
		    "audio input hardware\n");
	    return false;
	}
	if (!alsaio_init_input(alsa_fs_in,
			       fragsize * in_sf->bytes * n_channels /
			       alsa_fs_in, alsa_card_in,
			       alsa_dev_in, n_channels, rate, in_sf))
	{
	    fprintf(stderr, "Could not open ALSA device for input\n");
	    return false;
	}
	io_func->fillbuf = alsaio_input;
#else
	fprintf(stderr, "ALSA support not included in this binary\n");
	return false;
#endif
    } else {
	input_size = fragsize * in_sf->bytes * n_channels;
	io_func->fillbuf = stdfillbuf;
    }
    return true;
}

