/*
 * (c) Copyright 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#ifndef SAMPLE_H_
#define SAMPLE_H_

#include <inttypes.h>

#include "swap.h"
#include "defs.h"

struct sample_format {
    int bits;
    int bytes;
    int left_shift;
    bool_t is_little_endian;
    bool_t is_signed;
    /* to speed up to/from host format */
    bool_t swap;
    int shift;
};

/*
 * Parse sample format string:
 * <bits><s(igned)/u(nsigned)><bytes><l(ittle)/b(ig) endian><left shift>
 * examples: 16s2l0, 24u4b8
 */
bool_t
sample_parse_format(struct sample_format *sf,
		    char *s);

/*
 * Convert a raw sample to internal format, which is a signed 32 bit integer.
 * Sample values are scaled to be 32 bit.
 */
static inline int32_t
sample_raw2int(struct sample_format *sf,
	       void *p)
{
    uint32_t sample;
    
    switch (sf->bytes) {
    case 1:
	sample = 0;
	((uint8_t *)sample)[0] = *(uint8_t *)p;
	break;
    case 2:
	sample = 0;
	((uint16_t *)&sample)[0] = *(uint16_t *)p;
	break;	
    case 3:
	sample = 0;
	((uint8_t *)&sample)[0] = ((uint8_t *)p)[0];
	((uint8_t *)&sample)[1] = ((uint8_t *)p)[1];
	((uint8_t *)&sample)[2] = ((uint8_t *)p)[2];
	break;	
    case 4:
	sample = *(uint32_t *)p;
	break;
	
    }
    if (sf->swap) {
	switch (sf->bytes) {
	case 2:
	    sample = SWAP16(sample);
	    if (sample & 0x00008000 && sf->is_signed) {
		sample = sample | 0xFFFF0000;
	    }
	    break;
	case 3:
	    sample = SWAP32(sample);
	    if (sample & 0x00800000 && sf->is_signed) {
		sample = sample | 0xFF000000;
	    }
	    break;
	case 4:
	    sample = SWAP32(sample);
	    break;
	}
    }
    sample <<= sf->shift;
    if (!sf->is_signed) {
	sample += 0x80000000;
    }

    return sample;
}

/*
 * Convert a sample of internal format to raw format. The 'sample' variable must
 * have been reduced to fit into 'sf->bits' prior to calling this function.
 * The 'p' pointer is assumed to be aligned to 'sf->bytes'.
 */
static inline void
sample_int2raw(struct sample_format *sf,	       
	       void *p,
	       int32_t sample)
{
    if (!sf->is_signed) {
	(uint32_t)sample -= (1 << (sf->bits - 1));
    }
    sample <<= sf->left_shift;
    if (sf->swap) {
	sample = (sf->bytes == 2) ? SWAP16(sample) : SWAP32(sample);
    }
    switch (sf->bytes) {
    case 1:
	((uint8_t *)p)[0] = ((uint8_t *)&sample)[0];
	break;
    case 2:
	((uint16_t *)p)[0] = ((uint16_t *)&sample)[0];
	break;	
    case 3:
	((uint8_t *)p)[0] = ((uint8_t *)&sample)[0];
	((uint8_t *)p)[1] = ((uint8_t *)&sample)[1];
	((uint8_t *)p)[2] = ((uint8_t *)&sample)[2];
	break;	
    case 4:
	((uint32_t *)p)[0] = ((uint32_t *)&sample)[0];
	break;
	
    }
}

#endif
