/*
 * (c) Copyright 1999 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "defs.h"

/* linear interpolation */
bool_t
linear(double y_out[],
       double x_new[],
       double x[],
       double y[],
       int len,
       int len_new)
{
    int n, i;
    double a;

    if (len < 2) {
	fprintf(stderr, "linear: less than two input coordinates\n");
	return false;
    }
    if (x[0] > x_new[0] || x_new[len_new-1] > x[len-1]) {
	fprintf(stderr, "linear: x coordinates range mismatch (%f %f %f %f)\n",
		x[0], x_new[0], x[len-1], x_new[len_new-1]);
	return false;
    }

    a = (y[1] - y[0]) / (x[1] - x[0]);
    for (n = 0, i = 1; n < len_new; n++) {
	if (x_new[n] > x[i]) {
	    while (x_new[n] > x[i]) {
		i++;
	    }
	    a = (y[i] - y[i-1]) / (x[i] - x[i-1]);
	}
	if (x_new[n] == x[i]) {
	    y_out[n] = y[i];
	} else {
	    y_out[n] = y[i-1] + a * (x_new[n] - x[i-1]);
	}
    }
    return true;
}

/* spline interpolation */
bool_t
spline(double y_out[],
       double x_new[],
       double x[],
       double y[],
       int len,
       int len_new)
{
    double *dx = NULL, *dd = NULL, *b = NULL, *buf = NULL, *a[3], *c1, *c0;
    double xb, xe, tmp;
    int n, i, j, k = 0;
    
    if (len < 4) {
	fprintf(stderr, "spline: less than four input coordinates\n");
	return false;
    }
    dx = (double *)malloc(len * sizeof(double));
    dd = (double *)malloc(len * sizeof(double));
    b = (double *)malloc(len * sizeof(double));
    buf = (double *)malloc(3 * len * sizeof(double));
    if (dd == NULL || dx == NULL || buf == NULL || b == NULL) {
	fprintf(stderr, "spline: could not allocate memory\n");
	free(dx); free(dd); free(b); free(buf);
	return false;
    }
    dx[0] = x[1] - x[0];
    dd[0] = (y[1] - y[0]) / dx[0];
    for (n = 1; n < len - 1; n++) {
	dx[n] = x[n+1] - x[n];
	dd[n] = (y[n+1] - y[n]) / dx[n];
	b[n] = 3 * (dx[n] * dd[n-1] + dx[n-1] * dd[n]);
    }
    
    xb = x[2] - x[0];
    xe = x[len-1] - x[len-3];
    b[0] = ((dx[0] + 2 * xb) * dx[1] * dd[0] + dx[0] * dx[0] *
	    dd[1]) / xb;
    b[len-1] = (dx[len-2] * dx[len-2] * dd[len-3] + (2 * xe + dx[len-2]) *
		dx[len-3] * dd[len-2]) / xe;

    /* The Gaussian elimination step. a[0 .. 3] represents the sparse matrix.
       a[1] is the diagonal, a[0] contains the values below the diagonal,
       a[2] above. The rest is zeroes. Below an example of a 3 x 3 matrix.
       a[2][0] and a[0][len-1] are outside the matrix.
       
       a[1][0] a[2][1] 0

       a[0][0] a[1][1] a[2][2]

       0       a[0][1] a[1][2]
     */

    /* fill the matrix with values */
    a[0] = &buf[0];
    a[1] = &buf[len];
    a[2] = &buf[2 * len];    
    for (n = 0; n < len; n++) {
	a[0][n] = dx[n+1];
	a[1][n] = 2 * (dx[n] + dx[n-1]);
	a[2][n] = dx[n-2];
    }    
    a[0][len-2] = xe;
    a[0][len-1] = 0;
    a[1][0] = dx[1];
    a[1][len-1] = dx[len-3];
    a[2][0] = 0;
    a[2][1] = xb;
    
    /* perform Gaussian elimination, result stored in b[0 .. len - 1] */
    for (n = 1; n < len; n++) {
	a[1][n] = a[1][n] - a[0][n-1] / a[1][n-1] * a[2][n];
	b[n] = b[n] - a[0][n-1] / a[1][n-1] * b[n-1];
    }
    for (n = len - 1; n > 0; n--) {
	b[n-1] = b[n-1] - b[n] * a[2][n] / a[1][n];
	b[n] = b[n] / a[1][n];
    }
    b[0] = b[0] / a[1][0];

    /* calculate coefficients */
    c1 = &buf[0];
    c0 = &buf[len - 1];
    for (n = 0; n < len - 1; n++) {
	c0[n] = (b[n] + b[n+1] - 2 * dd[n]) / dx[n];
	c1[n] = (dd[n] - b[n]) / dx[n] - c0[n];
	c0[n] /= dx[n];
    }

    /* calculate output values */
    j = (x[0] <= x_new[0]) ? 1 : 0;
    for (n = 0, i = 0; n < len_new; n++) {
	if (x_new[n] >= x[i] && i < len - 1) {
	    i++;
	    k = i - j;
	}
	tmp = x_new[n] - x[k];
	y_out[n] = c0[k];
	y_out[n] = tmp * y_out[n] + c1[k];
	y_out[n] = tmp * y_out[n] + b[k];
	y_out[n] = tmp * y_out[n] + y[k];
    }
    free(buf);
    return true;
}
