/*
 * (c) Copyright 1999, 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <gsl/gsl_fft_real.h>
#include <gsl/gsl_fft_complex.h>
#include <gsl/gsl_fft_halfcomplex.h>

#include "defs.h"
#include "toolbox.h"
#include "spline.h"

/* This module contains various tools for filter design */

gsl_complex_packed_array
tb_hilbert(double x[],
	   int len)
{
    int pow, n;
    gsl_complex_packed_array complex;
  
    if ((pow = tb_getlog2(len)) < 2) {
	fprintf(stderr, "tb_hilbert: length not power of two or less than 4\n");
	return NULL;
    }
    if ((complex = malloc(len * sizeof(gsl_complex))) == NULL)
    {
	fprintf(stderr, "tb_hilbert: could not allocate memory\n");
	return NULL;
    }
    for (n = 0; n < len; n++) {
	complex[2*n] = x[n];
	complex[2*n+1] = 0;
    }
    gsl_fft_complex_radix2_forward(complex, 1, len);
    for (n = 1; n < len/2; n++) {
	complex[2*n] = 2 * complex[2*n];
	complex[2*n+1] = 2 * complex[2*n+1];
    }
    for (n = len/2+1; n < len; n++) {
	complex[2*n] = complex[2*n+1] = 0;
    }
    gsl_fft_complex_radix2_inverse(complex, 1, len);    
    return complex;
}

double *
tb_levinson_durbin(double x[],
		   int len,
		   int order)
{
    int n, i, k;
    double *y, *y1, a, b, c;

    if (len <= order) {
	fprintf(stderr, "tb_levinson_durbin: correlation vector is "
		"too short (%d %d)\n", len, order);
	return NULL;
    }
    
    y = malloc(order * sizeof(double));
    y1 = malloc((order + 1) * sizeof(double));
    y[0] = x[1] / x[0];
    for (n = 1; n < order; n++) {
	for (i = n, k = 0, a = 0, b = 0; i > 0; i--, k++) {
	    a += x[i] * y[k];
	    y1[k] = y[n-1-k];
	    b += x[i] * y1[k];
	}
	c = (x[n+1] - a) / (x[0] - b);
	for (i = 0; i < n; i++) {
	    y[i] = y[i] - c * y1[i];
	}
	y[i] = c;
    }
    y1[0] = 1;
    for (i = 1; i < order + 1; i++) {
	y1[i] = -y[i - 1];
    }
    free(y);
    
    return y1;
}

double *
tb_levinson(double x[],
	    int len,
	    double y[],
	    int order)
{
    double q, z;
    double *y1, *a, *a1, *buf;
    int n, i;

    y1 = (double *)malloc(order * sizeof(double));
    a = (double *)malloc(order * sizeof(double));
    a1 = (double *)malloc(order * sizeof(double));
    
    if (y1 == NULL || a == NULL || a1 == NULL) {
	fprintf(stderr, "tb_levinson: could not allocate memory\n");
	free(y1); free(a); free(a1);
	return NULL;
    }
    a[0] = 1;
    y1[0] = y[0] / x[0];
    q = x[0];
    for (n = 1; n < order; n++) {
	for (i = 1, z = 0; i <= n; i++) {
	    z -= a[n-i] * x[i];
	}
	z /= q;
	a1[0] = a[0];
	for (i = 1; i <= n; i++) {
	    a1[i] = a[i] + z * a[n-i];
	}
	buf = a;
	a = a1;
	a1 = buf;
	q *= (1 - fabs(z) * fabs(z));
	for (i = 1, z = 0; i <= n; i++) {
	    z += y1[n-i] * x[i];
	}
	z = (y[n] - z) / q;
	for (i = 0, y1[n] = 0; i <= n; i++) {
	    y1[i] += z * a[n-i];
	}
    }
    free(a1);
    free(a);
    return y1;
}

double * /* return array is of length 2 * len - 1 */
tb_cross_correlation(double x[],
		     int len)
{
    int nfft = 2;
    int n1 = 1, n2 = 1, n, i;
    double *af, *af_old, *c;
    int minus;

    while (nfft < 2 * len) nfft *= 2;    
    c = malloc(nfft * sizeof(double));
    af = malloc(nfft * sizeof(double));
    af_old = malloc(nfft * sizeof(double));
    if (c == NULL || af == NULL || af_old == NULL) {
	free(c); free(af); free(af_old);
	fprintf(stderr, "tb_cross_correlation: could not allocate memory\n");
	return NULL;
    }
    n1 = 0;
    bzero(c, 2 * len * sizeof(double));
    bzero(af_old, 2 * len * sizeof(double));
    while (n1 < len - 1) {
	n2 = (n1 + nfft / 2  < len) ? n1 + nfft / 2 : len;
	memcpy(af, &x[n1], n2 * sizeof(double));
	bzero(&af[n2], (2 * len - n2) * sizeof(double));
	gsl_fft_real_radix2_transform(af, 1, nfft);
	c[0] = c[0] + af[0] * af[0] + af_old[0];
	c[nfft/2] = c[nfft/2] + af[nfft/2] * af[nfft/2] + af_old[nfft/2];
	af_old[0] = af[0];
	af_old[nfft/2] = af[nfft/2];
	for (n = 1, minus = -1; n < nfft / 2; n++) {
	    c[n] = c[n] + af[n] * af[n] + af[nfft - n] * af[nfft - n] +
		af_old[n];
	    af_old[n] = af[n] * minus;
	    af_old[nfft - n] = af[nfft - n] * minus;
	    minus = minus * minus;
	}
	n1 += nfft / 2;
    }
    gsl_fft_halfcomplex_radix2_inverse(c, 1, nfft);
    for (n = 0, i = len - 1; n < len + 1; n++, i++) {
	af[i] = c[n];
    }    
    for (n = nfft - len + 1, i = 0; n < nfft; n++, i++) {
	af[i] = c[n];
    }
    free(c);
    free(af_old);
    return af;
}

double *
tb_log2lengthen(int *len,
		double signal[])
{
    int order = 0, n, i;
    double *output;

    if (tb_getlog2(*len) == -1) {
	for (n = 0, i = *len; n < 32; i >>= 1, n++) {
	    if ((i & 1) != 0) {
		order = n;
	    }
	}
	order++;
    } else {
	order = tb_getlog2(*len);
    }

    if ((output = malloc((1 << order) * sizeof(double))) == NULL) {
	fprintf(stderr, "tb_log2lengthen: could not allocate memory\n");
	return NULL;
    }
    bzero(output + *len, ((1 << order) - *len) * sizeof(double));
    memcpy(output, signal, *len * sizeof(double));    
    *len = (1 << order);
    return output;
}

void
tb_dbconvert(double signal[],
	     int len,
	     bool_t to_db)
{
    int n;

    if (to_db) {
	for (n = 0; n < len; n++) {
	    signal[n] = 20 * log10(signal[n]);
	}
    } else {
	for (n = 0; n < len; n++) {
	    signal[n] = pow(10, signal[n] / 20);
	}
    }
}

void
tb_degreeconvert(double signal[],
		 int len,
		 bool_t to_degrees)
{
    double factor;
    int n;

    if (to_degrees) {
	factor = 180.0 / M_PI;
    } else {
	factor = M_PI / 180.0;
    }
    for (n = 0; n < len; n++) {
	signal[n] *= factor;
    }
}

int
tb_getlog2(int x)
{
    int n;
    
    if (x < 1) {
	return -1;
    }
    for (n = 0; (x & 1) == 0 && n < 32; x = x >> 1, n++);
    if (n == 32 || (x & ~1) != 0) {
	return -1;
    }
    return n;
}


void
tb_abs_on_halfcomplex(double signal[],
		      int len)
{
    int n;
    
    for (n = 1; n < len / 2; n++) {
	signal[n] = sqrt(signal[n] * signal[n] +
			 signal[len-n] * signal[len-n]);
	signal[len-n] = signal[n];
    }
    signal[0] = fabs(signal[0]);
    signal[len/2] = fabs(signal[len/2]);
}

void
tb_halfcomplex2complex(gsl_complex_packed_array complex,
		       double halfcomplex[],		       
		       int len)
{
    int n;
    complex[0] = halfcomplex[0];
    complex[1] = 0;
    for (n = 1; n < len/2; n++) {
	complex[2*n] = halfcomplex[n];
	complex[2*n+1] = halfcomplex[len-n];
	complex[2*(len-n)] = halfcomplex[n];
	complex[2*(len-n)+1] = -halfcomplex[len-n];
    }
    /* 001001; is this really correct? It doesn't seem like it */
    complex[2*len] = halfcomplex[len/2];
    complex[2*len+1] = 0;
}

double *
tb_interpolate(double **x_new,
	       double x[],
	       double y[],
	       int len,
	       int newlen,
	       bool_t set_endpoints,
	       double first_x,
	       double last_x,
	       double first_y,
	       double last_y)
{
    double *x1 = NULL, *y1 = NULL, *xnew = NULL, *ynew = NULL;
    int len1, n;

    if (len < 3) {
	fprintf(stderr, "tb_interpolate: input length must at least be 2\n");
    }
    
    if ((x1 = malloc((len + 2) * sizeof(double))) == NULL ||
	(y1 = malloc((len + 2) * sizeof(double))) == NULL ||
	(xnew = malloc(newlen * sizeof(double))) == NULL ||
	(ynew = malloc(newlen * sizeof(double))) == NULL)
    {
	fprintf(stderr, "tb_interpolate: memory allocation failure\n");
	free(x1); free(y1); free(xnew); free(ynew);
	return NULL;
    }
    if (set_endpoints) {
	xnew[0] = first_x;
	for (n = 1; n < newlen-1; n++) {
	    xnew[n] = first_x +
		(double)n / (double)(newlen - 1) * (last_x - first_x);
	}
	xnew[newlen-1] = last_x;
	if (x[0] != first_x) {
	    memcpy(&x1[1], x, len * sizeof(double));
	    memcpy(&y1[1], y, len * sizeof(double));	    
	    x1[0] = first_x;
	    len1 = len + 1;
	} else {
	    memcpy(x1, x, len * sizeof(double));
	    memcpy(y1, y, len * sizeof(double));
	    len1 = len;
	}
	if (x[len-1] != last_x) {
	    x1[len1] = last_x;
	    len1++;
	}
	y1[0] = first_y;;
	y1[len1-1] = last_y;
    } else {
	xnew[0] = x[0];
	for (n = 1; n < newlen-1; n++) {
	    xnew[n] = x[0] +
		(double)n / (double)(newlen - 1) * (x[len-1] - x[0]);
	}
	xnew[newlen-1] = x[len-1];
	memcpy(x1, x, len * sizeof(double));
	memcpy(y1, y, len * sizeof(double));
	len1 = len;
    }
    if (!linear(ynew, xnew, x1, y1, len1, newlen)) {
	fprintf(stderr, "linear_phase_filter: linear interpolation failed\n");
	free(x1); free(y1); free(xnew); free(ynew);
	return NULL;
    }
    free(x1); free(y1);
    if (x_new != NULL) {
	*x_new = xnew;
    } else {
	free(xnew);
    }
    return ynew;
}

double *
tb_splinewarp(double signal[],
	      int len,
	      double warp_factor,
	      double sfreq)
{
    double *fb = malloc(len * sizeof(double));
    double *fbw = malloc(len * sizeof(double));
    int n;

    if (fb == NULL || fbw == NULL) {
	fprintf(stderr, "tb_splinewarp: could not allocate memory\n");
	free(fb); free(fbw);
	return NULL;
    }
    
    if (warp_factor == 0) {
	free(fbw);
	memcpy(fb, signal, len * sizeof(double));
	return fb;
    }

    /* Equidistant frequency spacing */
    for (n = 1; n <= len; n++) {
	fb[n-1] = sfreq / 2 * n / len;
    }

    /* Convert to warped frequency spacing */
    for (n = 0; n < len; n++) {
	fbw[n] = fb[n] + (sfreq / M_PI) *
	    atan(warp_factor * sin(2 * M_PI * fb[n] / sfreq) /
		 (1.0 - warp_factor * cos(2 * M_PI * fb[n] / sfreq)));
    }

    /* Calculate warped signal, with spline approximation */
    if (!spline(fb, fb, fbw, signal, len, len)) {
	free(fb); free(fbw);
	fprintf(stderr, "tb_splinewarp: spline interpolation failed\n");
	return NULL;
    }
    free(fbw);
    return fb;
}

double *
tb_linearwarp(double signal[],
	      int len,
	      double warp_factor,
	      double sfreq)
{
    double *fb = malloc(len * sizeof(double));
    double *fbw = malloc(len * sizeof(double));
    int n;

    if (fb == NULL || fbw == NULL) {
	fprintf(stderr, "tb_linearwarp: could not allocate memory\n");
	free(fb); free(fbw);
	return NULL;
    }

    if (warp_factor == 0) {
	free(fbw);
	memcpy(fb, signal, len * sizeof(double));
	return fb;
    }

    /* Equidistant frequency spacing */
    for (n = 1; n <= len; n++) {
	fb[n-1] = sfreq / 2 * n / len;
    }

    /* Convert to warped frequency spacing */
    for (n = 0; n < len; n++) {
	fbw[n] = fb[n] + (sfreq / M_PI) *
	    atan(warp_factor * sin(2 * M_PI * fb[n] / sfreq) /
		 (1.0 - warp_factor * cos(2 * M_PI * fb[n] / sfreq)));
    }
    /* Make sure that fb is within fbw's range. This will induce a small error,
       but is needed for the linear approximation to work. */
    if (fbw[0] > fb[0]) {
	fbw[0] = fb[0];
    }
    if (fb[len-1] > fbw[len-1]) {
	fbw[len-1] = fb[len-1];
    }
    
    /* Calculate warped signal, with linear approximation */
    if (!linear(fb, fb, fbw, signal, len, len)) {
	free(fb); free(fbw);
	fprintf(stderr, "tb_linearwarp: linear interpolation failed\n");
	return NULL;
    }
    free(fbw);
    return fb;
}

bool_t
tb_real_cepstrum(double rc[], /* output: real cepstrum of x */
		 double ep[], /* output: excess phase signal of x */
		 double mp[], /* output: minimum phase signal of x */
		 gsl_complex_packed_array x,  /* input signal */
		 int len)  
{
    int n;
    double a, b, c;
    gsl_complex_packed_array complex = NULL, complex2 = NULL;

    if (tb_getlog2(len) < 2 || rc == NULL || mp == NULL || x == NULL) {
	fprintf(stderr, "tb_real_cepstrum: at least one invalid input "
		"parameter\n");
	return false;
    }
    
    /* calculate real cepstrum */
    if ((complex = malloc(len * sizeof(gsl_complex))) == NULL ||
	(complex2 = malloc(len * sizeof(gsl_complex))) == NULL)
    {
	fprintf(stderr, "tb_real_cepstrum: could not allocate memory\n");
	return false;
    }
    memcpy(complex, x, len * sizeof(gsl_complex));
    gsl_fft_complex_radix2_forward(complex, 1, len);
    memcpy(complex2, complex, len * sizeof(gsl_complex));
    for (n = 0; n < len; n++) {
	complex[2*n] = log(sqrt(complex[2*n] * complex[2*n] +
				complex[2*n+1] * complex[2*n+1]));
	complex[2*n+1] = 0;
    }
    gsl_fft_complex_radix2_inverse(complex, 1, len);
    for (n = 0; n < len; n++) {
	rc[n] = complex[2*n];
    }
    free(complex);
        
    /* calculate minimum/excess phase signal */
    mp[0] = rc[0];
    for (n = 1; n < len / 2; n++) {
	mp[n] = 2 * rc[n];
    }
    mp[n] = rc[n];
    bzero(&mp[len/2+1], (len / 2 - 1) * sizeof(double));
    gsl_fft_real_radix2_transform(mp, 1, len);
    mp[0] = exp(mp[0]);
    mp[len/2] = exp(mp[len/2]);
    for (n = 1; n < len / 2; n++) {
	a = exp(mp[n]);
	mp[n] = a * cos(mp[len-n]);
	mp[len-n] = a * sin(mp[len-n]);
    }
    /*
     * Before fft inverting to get the minimum phase signal, we use the current 
     * result to calculate the excess phase signal.
     */
    ep[0] = complex2[0] / mp[0];
    ep[len/2] = complex2[2*len] / mp[len/2];
    for (n = 1; n < len / 2; n++) {
	a = mp[n] * mp[n] + mp[len-n] * mp[len-n];
	b = complex2[2*n] * mp[n] + complex2[2*n+1] * mp[len-n];
	c = complex2[2*n+1] * mp[n] - complex2[2*n] * mp[len-n];
	ep[n] = b / a;
	ep[len-n] = c / a;
    }
    free(complex2);
    gsl_fft_halfcomplex_radix2_inverse(ep, 1, len);
    gsl_fft_halfcomplex_radix2_inverse(mp, 1, len);
    return true;
}

bool_t
tb_hilbert_phase_split(double ep[],
		       double mp[],
		       double x[],
		       int len)
{
    gsl_complex_packed_array complex, complex2;
    double *a, arg;
    int n;

    if (ep != NULL) {
	fprintf(stderr, "tb_hilbert_phase_split: excess phase extraction not "
		"implemented\n");
	return false;
    }
    
    if (tb_getlog2(len) < 2) {
	fprintf(stderr, "tb_hilbert_phase_split: length not power of two or "
		"less than 4\n");
	return false;
    }
    
    if ((a = malloc(len * sizeof(double))) == NULL ||
	(complex2 = malloc(len * sizeof(gsl_complex))) == NULL)
    {
	fprintf(stderr, "tb_hilbert_phase_split: could not allocate memory\n");
	return false;
    }
    memcpy(a, x, len * sizeof(double));
    
    gsl_fft_real_radix2_transform(a, 1, len);
    tb_abs_on_halfcomplex(a, len);
    if ((complex = tb_hilbert(a, len)) == NULL) {
	return false;
    }
    memcpy(complex2, complex, len * sizeof(gsl_complex));
    for (n = 0; n < len; n++){
	arg = atan2(complex[2*n+1], complex[2*n]);
	complex2[2*n] = cos(-arg) * a[n];
	complex2[2*n+1] = sin(-arg) * a[n];
    }
    gsl_fft_complex_radix2_inverse(complex2, 1, len);
    for (n = 0; n < len; n++){
	mp[n] = complex2[2*n];
    }
    
    return true;
}

double *
tb_linear_predictor_coefficients(double signal[],
				 int len,
				 int order)
{
    double *a, *b;

    a = tb_cross_correlation(signal, len);
    b = tb_levinson_durbin(&a[len - 1], len, order);
    free(a);
    return b;    
}
