/*
 * (c) Copyright 1999, 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU Public License as published by the Free Software Foundation.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <math.h>

#include "wfir.h"
#include "coeffs.h"
#include "timestamp.h"

/* structure to WFIR filter data needed during calculations */
struct wfirdata {
    int32_t *state;     /* buffer storing the internal state of the filter */
    int32_t *coeffs;    /* filter coefficients as upscaled integers */
    int32_t multiplier;
    int taps;           /* length of the filter (state and coeffs buffers) */
    bool_t active;      /* true if filter calculation is active */
    /* pointer to one of the wfir_process_asm* functions */
    int64_t (*process)(int32_t, int32_t *, int, int32_t *); 
};

/* toggle filter processing */
static void
toggle_processing(struct wfirdata *wfirdata)
{
    wfirdata->active = !wfirdata->active;
}

/*
 * Process a single sample.
 */
static int64_t
process_sample(struct wfirdata *wfirdata,
	       int32_t sample)
{
    int64_t a;

    sample >>= 8;
    if (wfirdata->active) {
	a = wfirdata->process(sample, wfirdata->coeffs, wfirdata->taps,
			      wfirdata->state);
    } else {
	a = sample;
	a *= wfirdata->multiplier;
    }
    return a;
}

/*
 * c implementation of the wfir_process_asm function found in wfirloop.s.
 * Used only for reference.
 */
int64_t
wfir_process_int(int32_t sample,
		 int32_t coeffs[],
		 int taps,
		 int32_t state[])
{
    int64_t y, b;
    int32_t z, buf;
    int n;

    for (n = 0, z = sample, y = 0; n < taps; n++) {
	buf = state[n] + state[n + 1] - z - (state[n + 1] - z) / 128;
	state[n] = z;
	z = buf;
	b = state[n];
	y += b * coeffs[n];
    }
    return y;    
}

/*
 * Real valued version of wfir_process_asm that acts on unscaled samples and
 * coefficients. Used only for reference.
 */
double
wfir_process_real(double sample,
		  double coeffs[],
		  int taps,
		  double state[],
		  double warp_factor)
{
    int n;
    double z, buf, y;

    for(n = 0, z = sample, y = 0; n < taps; n++) {
	/* buf = state[n] + state[n + 1] - z - (state[n + 1] - z) / 128.0; */
	buf = state[n] + warp_factor * (state[n + 1] - z);	   
	state[n] = z;
	z = buf;
	y += coeffs[n] * state[n];
    }
    return y;
}

/*
 * Create a new WFIR filter, and return it in the generalised filterproc_t
 * type. The three last parameters receives only informational values. The
 * pointers may be set to NULL.
 */
filterproc_t *
wfir_new(char coeffs_filename[], /* file containing the filter coefficients */
	 double warp_factor,     /* warp factor, one of 1/2, 3/4 ... 255/256 */
	 double gain,
	 int *taps,              /* receives the filter length */
	 double *coeffs_error)   /* receives the coeff requantisation error */
{
    filterproc_t *filter = malloc(sizeof(filterproc_t));
    struct wfirdata *wfirdata = malloc(sizeof(struct wfirdata));
    FILE *stream;
    double **coeffs;

    if (filter == NULL || wfirdata == NULL) {
	free(filter); free(wfirdata);	
	fprintf(stderr, "wfir_new: Could not allocate memory\n");
	return NULL;
    }
    if (warp_factor == 1.0 / 2.0) {
	wfirdata->process = wfir_process_asm1;
    } else if (warp_factor == 3.0 / 4.0) {
	wfirdata->process = wfir_process_asm3;
    } else if (warp_factor == 7.0 / 8.0) {
	wfirdata->process = wfir_process_asm7;
    } else if (warp_factor == 15.0 / 16.0) {
	wfirdata->process = wfir_process_asm15;
    } else if (warp_factor == 31.0 / 32.0) {
	wfirdata->process = wfir_process_asm31;
    } else if (warp_factor == 63.0 / 64.0) {
	wfirdata->process = wfir_process_asm63;
    } else if (warp_factor == 127.0 / 128.0) {
	wfirdata->process = wfir_process_asm127;
    } else if (warp_factor == 255.0 / 256.0) {
	wfirdata->process = wfir_process_asm255;
    } else {
	fprintf(stderr, "wfir_new: Invalid warp factor: %f\n", warp_factor);
	free(filter); free(wfirdata);	
	return NULL;
    }

    if ((stream = fopen(coeffs_filename, "rt")) == NULL) {
	fprintf(stderr, "wfir_new: Could not open file \"%s\"\n",
		coeffs_filename);
	free(filter); free(wfirdata);	
	return NULL;
    }
    if ((coeffs = coeffs_parse(&wfirdata->taps, stream)) == NULL) {
	fprintf(stderr, "wfir_new: Failed to parse coefficients\n");
	free(filter); free(wfirdata);	
	return NULL;
    }
    fclose(stream);
    if (taps != NULL) {
	*taps = wfirdata->taps;
    }
    wfirdata->coeffs = coeffs_requantisise(coeffs[0],
					   wfirdata->taps,
					   24,
					   &wfirdata->multiplier,
					   coeffs_error);
    coeffs_free(coeffs);
    if (wfirdata->coeffs == NULL) {
	fprintf(stderr, "wfir_new: Coefficient requantisation failed\n");
	free(filter); free(wfirdata);
	return NULL;
    }
    
    if ((wfirdata->state = malloc(wfirdata->taps * sizeof(int32_t))) == NULL) {
	fprintf(stderr, "wfir_new: Could not allocate memory\n");
	free(filter); free(wfirdata);	
	return NULL;	
    }
    bzero(wfirdata->state, *taps * sizeof(int32_t));
    wfirdata->active = true;

    filter->process_sample = (int64_t (*)(void *, int32_t))process_sample;
    filter->toggle_processing = (void (*)(void *))toggle_processing;
    filter->private = (void *)wfirdata;
    filter->multiplier = wfirdata->multiplier >> 8;
    filter->db_gain = gain;
    
    return filter;
}

