/*
 * (c) Copyright 1999, 2000 -- Anders Torger
 *
 * This software is free. You can redistribute it and/or modify it under the
 * terms of the GNU Public License as published by the Free Software Foundation.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <strings.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <gsl/gsl_fft_real.h>
#include <gsl/gsl_fft_complex.h>
#include <gsl/gsl_fft_halfcomplex.h>

#include "defs.h"
#include "spline.h"
#include "toolbox.h"
#include "coeffs.h"

#define DEFAULT_WARP 0.9921875
#define DEFAULT_FFTSIZE 32768
#define DEFAULT_FREQ 44100

#define PRESENTATION_STRING \
"\n"\
"W F I R D -- real valued wfir design                       Experimental version\n"\
"====================================                          (c) Anders Torger\n"\
"                                                                      June 2000\n"\
"\n"

#define USAGE_STRING \
"Usage: %s [-ior] [-d <output option>] [-f <fft size>] [-g <gain in dB>]\n"\
"       [-s <sample frequency>] [-w <warp factor>] \n"\
"       <target power response file> <number of taps>\n"\
"\n"\
" -i power response input is in decibel\n"\
" -o output in decibel\n"\
" -r output memory images of the values rather than human readable form\n"\
" -d <output option> selects what to output:\n"\
"    COEFFS    -- filter coefficients (default)\n"\
"    TARGET    -- target response (interpolated)\n"\
"    FILTER    -- filter response\n"\
"    PHASE     -- filter phase response\n"\
"    WTARGET   -- warped target response\n"\
"    WFILTER   -- warped filter response\n"\
"    ERROR     -- difference between target and filter response\n"\
"    ALL <dir> -- output all of the above to the directory given.\n"\
"                 Filenames are coeffs.dat, target.dat, etc.\n"\
" -f <fft size> Fast Fourier Transform working set size. Indirectly specifies\n"\
"    the frequency resolution which is <sample frequency> / <fft size>.\n"\
"    <fft size> must be a power of two. Default: 32768\n"\
" -g <gain in dB> filter gain in decibel. Default: auto gain\n"\
" -s <sample frequency> default value: 44100\n"\
" -w <warp factor> default value: 0.9921875 (which is 127/128)\n"\
"\n"

#define OUTPUT_COEFFS   1
#define OUTPUT_TARGET   2
#define OUTPUT_FILTER   3
#define OUTPUT_PHASE    4
#define OUTPUT_WTARGET  5
#define OUTPUT_WFILTER  6
#define OUTPUT_ERROR    7
#define OUTPUT_ALL      8

/* 001001: This code needs some cleanup, it's quite inefficient */
static double *
design_wfir(double **wtarget,   /* receives the warped target power response */
	    double **wfilter,   /* receives the warped filter power response */
	    double **filter,    /* receives the filter power response */
	    double **phase,     /* receives the filter phase response */
	    double target[],    /* The target response, i.e. the power response
			         * at equidistant points 0+r Hz to sfreq/2+r Hz
			         * where r is sfreq/len/4 */
	    int len,            /* length of the target response, must be a
			         * power of two */
	    double warp_factor, /* warp factor, must be within -1 to 1 */
	    int n_taps,         /* number of filter taps */
	    double sfreq,       /* sample frequency */
	    double scale)       /* scale value for coefficients. 0 = undef */
{
    double *tmp, *lpc, *rc, *mp = NULL, *ep = NULL;
    gsl_complex_packed_array complex = NULL;
    int n, fftsize = 2 * len;

    /* do sanity check of the input parameters */
    if (target == NULL || tb_getlog2(len) == -1 || n_taps < 1 || sfreq <= 0) {
	fprintf(stderr, "design_wfir: at least one input parameter is "
		"invalid\n");
	return NULL;
    }
    if (warp_factor <= -1.0 || warp_factor >= 1.0) {
	fprintf(stderr, "design_wfir: invalid warp factor, must be "
		"-1.0 < wf < 1.0 for a stable filter\n");
	return NULL;
    }
    
    /* warp the target */
    if ((*wtarget = tb_linearwarp(target, len, warp_factor, sfreq)) == NULL) {
	fprintf(stderr, "design_wfir: prewarp failed\n");
	return NULL;
    }
    
    /* invert the warped target (and leave space for inverse fft) */
    if ((tmp = malloc(fftsize * sizeof(double))) == NULL) {
	free(*wtarget);
	fprintf(stderr, "design_wfir: could not allocate memory\n");
	return NULL;
    } 
    memcpy(tmp, *wtarget, (len + 1) * sizeof(double));
    bzero(&tmp[len+1], (len - 1) * sizeof(double));
    for (n = 0; n <= len; n++) {
	tmp[n] = 1 / tmp[n];
    }

    /* Inverse fft of target */
    if ((complex = malloc(2 * fftsize * sizeof(double))) == NULL) {
	free(*wtarget); free(tmp);
	fprintf(stderr, "design_wfir: could not allocate memory\n");
	return NULL;
    }
    tb_halfcomplex2complex(complex, tmp, fftsize);
    free(tmp);
    gsl_fft_complex_radix2_inverse(complex, 1, fftsize);
    /*
     * Get minimum phase. The minimum phase is a side-product of the real
     * cepstrum calculation. (We are not interested in the real cepstrum).
     */
    if ((rc = malloc(fftsize * sizeof(double))) == NULL ||
	(ep = malloc(fftsize * sizeof(double))) == NULL ||
	(mp = malloc(fftsize * sizeof(double))) == NULL)
    {
	free(*wtarget); free(rc); free(ep); free(mp); free(complex);
	fprintf(stderr, "design_wfir: could not allocate memory\n");
	return NULL;
    }
    for (n = 0; n < fftsize; n++) {
	complex[2 * n + 1] = 0;
    }
    if (!tb_real_cepstrum(rc, ep, mp, complex, fftsize)) {
	free(*wtarget); free(rc); free(mp); free(complex);
	fprintf(stderr, "design_wfir: could not calculate minimum phase\n");
	return NULL;
    }
    free(complex);
    free(ep);
    free(rc);
    
    /* find filter coefficients */
    lpc = tb_linear_predictor_coefficients(mp, fftsize, n_taps);
    free(mp);
    if (lpc == NULL) {
	free(*wtarget);
	fprintf(stderr, "design_wfir: lpc failed\n");
	return NULL;
    }

    /* calculate warped filter power and phase response */
    if ((*wfilter = malloc(fftsize * sizeof(double))) == NULL ||
	(tmp = malloc(len * sizeof(double))) == NULL)
    {
        free(*wtarget); free(*wfilter); free(tmp);
        fprintf(stderr, "design_wfir: could not allocate memory\n");
        return NULL;
    }
    bzero(*wfilter, fftsize * sizeof(double));
    memcpy(*wfilter, lpc, n_taps * sizeof(double));
    gsl_fft_real_radix2_transform(*wfilter, 1, fftsize);
    tmp[0] = 0;
    for (n = 1; n < fftsize / 2; n++) {
	tmp[n] = atan2((*wfilter)[fftsize-n], (*wfilter)[n]);
    }
    tmp[fftsize/2] = 0;
    tb_abs_on_halfcomplex(*wfilter, fftsize);
    /* rescale coefficents */    
    if (scale == 0) {
	/* auto scaling */
	if (warp_factor < 0) {
	    /* best precision in high frequencies */
	    for (n = fftsize - 1; n >= 3 * fftsize / 4; n--) {
		scale += (*wtarget)[n] / (*wfilter)[n];
	    }
	} else {
	    /* best precision in low frequencies */
	    for (n = 0; n < fftsize / 4; n++) {
		scale += (*wtarget)[n] / (*wfilter)[n];
	    }
	}
	scale /= (fftsize / 4);
    }
    for (n = 0; n < n_taps; n++) {
	lpc[n] = lpc[n] * scale;
    }
    
    for (n = 0; n < fftsize; n++) {
	(*wfilter)[n] = (*wfilter)[n] * scale;
    }
        
    /* calculate filter power and phase response */
    *filter = tb_linearwarp(*wfilter, len, -warp_factor, sfreq);
    *phase = tb_linearwarp(tmp, len, -warp_factor, sfreq);
    free(tmp);
    
    return lpc;
}

#define WRITE_COEFFS_PRINT_OUTPUT(len)                                         \
        if (!coeffs_print(stream, tmp, len, rawoutput)) {                      \
	    fprintf(stderr, "Could not write output\n");                       \
    	    return 1;                                                          \
        }

#define OUTPUT_STD_MACRO(signal, allow_convertdb)                              \
	if ((allow_convertdb) && dboutput) {                                   \
            tb_dbconvert(signal, fftsize / 2, true);                           \
	}                                                                      \
	tmp[1] = signal;                                                       \
	WRITE_COEFFS_PRINT_OUTPUT(fftsize / 2);

#define OPEN_STREAM(name)                                                      \
        if ((stream = fopen(name, "wt")) == NULL) {                            \
            fprintf(stderr, "failed to open file \"" name "\" for writing\n"); \
	    return 1;                                                          \
        }

int
main(int argc,
     char *argv[])
{
    double *coeffs, *wtarget, *target, *wfilter, *filter, *x, *tmp[3], *phase;
    double sfreq = DEFAULT_FREQ, **input, warp_factor = DEFAULT_WARP;
    double *targetcopy = NULL, *filtercopy = NULL, gain = 0;
    int len, n, taps, fftsize = DEFAULT_FFTSIZE, output_content = OUTPUT_COEFFS;
    bool_t dbinput = false, dboutput = false, rawoutput = false;
    FILE *stream;
    char output_dir[1024];
    
    fprintf(stderr, PRESENTATION_STRING);

    /* parse arguments */
    if (argc < 3) {
	fprintf(stderr, USAGE_STRING, argv[0]);
	return 1;
    }
    for (n = 1; n < argc - 2 && argv[n][0] == '-'; n++) {
	switch (argv[n][1]) {
	case 'i':
	    dbinput = true;
	    break;
	case 'o':
	    dboutput = true;
	    break;
	case 'r':
	    rawoutput = true;
	    break;
	case 'd':
	    n++;
	    if (strcasecmp(argv[n], "COEFFS") == 0) {
		output_content = OUTPUT_COEFFS;
	    } else if (strcasecmp(argv[n], "TARGET") == 0) {
		output_content = OUTPUT_TARGET;
	    } else if (strcasecmp(argv[n], "FILTER") == 0) {
		output_content = OUTPUT_FILTER;
	    } else if (strcasecmp(argv[n], "PHASE") == 0) {
		output_content = OUTPUT_PHASE;
	    } else if (strcasecmp(argv[n], "WTARGET") == 0) {
		output_content = OUTPUT_WTARGET;
	    } else if (strcasecmp(argv[n], "WFILTER") == 0) {
		output_content = OUTPUT_WFILTER;
	    } else if (strcasecmp(argv[n], "ERROR") == 0) {
		output_content = OUTPUT_ERROR;
	    } else if (strcasecmp(argv[n], "ALL") == 0) {
		n++;
		output_content = OUTPUT_ALL;
		strncpy(output_dir, argv[n], 1023);
		output_dir[1023] = '\0';
	    } else {
		fprintf(stderr, "Invalid output option\n");
		return 1;
	    }
	    break;
	case 'f':
	    fftsize = atoi(argv[++n]);
	    if (tb_getlog2(fftsize) == -1) {
		fprintf(stderr, "Invalid fft size \"%s\". Must be a power of "
			"two\n", argv[n]);
		return 1;
	    }
	    break;
	case 'g':
	    gain = pow(10.0, atof(argv[++n])/20.0);
	    break;
	case 's':
	    sfreq = atof(argv[++n]);
	    if (sfreq <= 0) {
		fprintf(stderr, "Invalid sample frequency \"%s\"\n", argv[n]);
		return 1;
	    }
	    break;
	case 'w':
	    warp_factor = atof(argv[++n]);
	    if (warp_factor <= -1 || warp_factor >= 1) {
		fprintf(stderr, "Invalid warp factor, must be within the range "
			"-1 to 1.\n");
		return 1;
	    }
	    break;
	default:
	    fprintf(stderr, "Invalid argument\n\n");
	    fprintf(stderr, USAGE_STRING, argv[0]);
	    return 1;
	}
    }
    if (n != argc - 2) {
	fprintf(stderr, "Missing parameter(s)\n");
	return 1;
    }
    if (output_content == OUTPUT_COEFFS && dboutput) {
	fprintf(stderr, "Coefficients in dB scale makes no sense. Remove -o "
		"parameter\n");
	return 1;
    }
    if (argc <= n + 1) {
	fprintf(stderr, "Missing arguments\n\n");
	fprintf(stderr, USAGE_STRING, argv[0]);
	return 1;
    }
    if ((taps = atoi(argv[n+1])) <= 0) {
	fprintf(stderr, "Invalid number of taps: \"%s\"\n", argv[n+1]);
	return 1;
    }
    if ((stream = fopen(argv[n], "rt")) == NULL) {
	fprintf(stderr, "Failed to open \"%s\" for reading\n", argv[n]);
	return 1;
    }
    
    /* read target response */
    if ((input = coeffs_parse(&len, stream)) == NULL) {
	fprintf(stderr, "Failed to parse power response file\n");
	return 1;
    }
    if (input[1] == NULL) {
	fprintf(stderr, "Frequency or magnitude column missing "
		"in power response file\n");
	return 1;
    }

    /* create linear interpolation of the target response */
    x = malloc(fftsize / 2 * sizeof(double));
    target = malloc(fftsize / 2 * sizeof(double));
    if (x == NULL || target == NULL) {
	fprintf(stderr, "Failed to allocate memory\n");
	return 1;
    }
    for (n = 1; n <= fftsize / 2; n++) {
	x[n-1] = sfreq * (double)n / fftsize;
    }
    if (input[0][0] > x[0]) {
	input[0][0] = x[0];
    }
    if (x[fftsize/2-1] > input[0][len-1]) {
	input[0][len-1] = x[fftsize/2-1];
    }
    if (!linear(target, x, input[0], input[1], len, fftsize / 2)) {
	fprintf(stderr, "Interpolation of target power response failed\n");
	return 1;
    }
    if (dbinput) {
	tb_dbconvert(target, fftsize / 2, false);	
    }

    /* do the filter design */
    if ((coeffs = design_wfir(&wtarget,
			      &wfilter,
			      &filter,
			      &phase,
			      target,
			      fftsize / 2,
			      warp_factor,
			      taps,
			      sfreq,
			      gain))
	== NULL)
    {
	fprintf(stderr, "Filter design failed\n");
	return 1;
    }

    /* print output */
    tmp[0] = x;
    tmp[2] = NULL;
    stream = stdout;    
    switch (output_content) {
    case OUTPUT_COEFFS:
	tmp[0] = coeffs;
	tmp[1] = NULL;
	WRITE_COEFFS_PRINT_OUTPUT(taps);
	break;
    case OUTPUT_TARGET:
	OUTPUT_STD_MACRO(target, true);
	break;
    case OUTPUT_FILTER:
	OUTPUT_STD_MACRO(filter, true);
	break;
    case OUTPUT_PHASE:
	OUTPUT_STD_MACRO(phase, false);
	break;
    case OUTPUT_WTARGET:
	OUTPUT_STD_MACRO(wtarget, true);
	break;
    case OUTPUT_WFILTER:
	OUTPUT_STD_MACRO(wfilter, true);
	break;
    case OUTPUT_ERROR:
	for (n = 0; n < fftsize / 2; n++) {
	    target[n] = filter[n] - target[n] + ((dboutput) ? 1 : 0);
	}
	OUTPUT_STD_MACRO(target, true);
	break;
    case OUTPUT_ALL:
	if (chdir(output_dir) == -1) {
	    fprintf(stderr, "Failed to enter directory \"%s\" (%s)\n",
		    output_dir, strerror(errno));
	    return 1;
	}
	if (dboutput) {
	    targetcopy = malloc(fftsize / 2 * sizeof(double));
	    filtercopy = malloc(fftsize / 2 * sizeof(double));
	    if (targetcopy == NULL || filtercopy == NULL) {
		fprintf(stderr, "Could not allocate memory\n");
		return 1;
	    }
	    memcpy(targetcopy, target, fftsize / 2 * sizeof(double));
	    memcpy(filtercopy, filter, fftsize / 2 * sizeof(double));
	}
	OPEN_STREAM("coeffs.dat");
	tmp[0] = coeffs;
	tmp[1] = NULL;
	WRITE_COEFFS_PRINT_OUTPUT(taps);
	fclose(stream);
	tmp[0] = x;
	
	OPEN_STREAM("target.dat");
	OUTPUT_STD_MACRO(target, true);
	fclose(stream);
	
	OPEN_STREAM("filter.dat");
	OUTPUT_STD_MACRO(filter, true);
	fclose(stream);
	
	OPEN_STREAM("phase.dat");
	OUTPUT_STD_MACRO(phase, false);
	fclose(stream);
	
	OPEN_STREAM("wtarget.dat");
	OUTPUT_STD_MACRO(wtarget, true);
	fclose(stream);
	
	OPEN_STREAM("wfilter.dat");
	OUTPUT_STD_MACRO(wfilter, true);
	fclose(stream);
	
	OPEN_STREAM("error.dat");
	if (dboutput) {
	    target = targetcopy;
	    filter = filtercopy;
	}
	for (n = 0; n < fftsize / 2; n++) {
	    target[n] = filter[n] - target[n] + ((dboutput) ? 1 : 0);
	}
	OUTPUT_STD_MACRO(target, true);
	fclose(stream);
	break;
    default:
	fprintf(stderr, "Internal error: invalid output content\n");
	return 1;
    }
    fprintf(stderr, "Filter design finished successfully\n");
    return 0;
}
